/**
 * \file: svg_font_freetype.c
 *
 * \version: $Id: svg_font_freetype.c,v 1.143.4.2.2.1.2.4.2.7 2012/11/14 08:18:45 harsha.mm Exp $
 *
 * \component:  svg_font_freetype for displaying texts with freetype library
 *
 * \author: d.langner ADIT.G
 *
 * \copyright: (c) 2003 - 2010 ADIT Corporation
 *
 * \history
 *
 ***********************************************************************/

#include "svg_typedef.h"
#include "grlft_font.h"
#include "grl_font.h"
#include "svg_error.h"
#include "svg_font_err.h"
#include "svg_math_lut.h"
#include "svg_font.h"
#include "grl_os_abstraction.h"
#include "grl_configuration.h"

#include <stdio.h>
#include <unistd.h>

#ifdef HAS_OPENVG
#include "VG/openvg.h"
#endif

#ifdef HAS_OPENGLES1
#ifdef HAS_OPENGLES2
#error ("do not include both versions of OpenGL!")
#endif
#include "GLES/gl.h"
#include "GLES/glext.h"
#endif

#ifdef HAS_OPENGLES2
#include "GLES2/gl2.h"
#include "GLES2/gl2ext.h"
#include "fragment.h"
#include "vertex.h"



static char const * const shaderFiles[2]=
{   "varying vec2 texcoord;\
     attribute vec4 position;\
     attribute vec2 inputtexcoord;\
     uniform mat4 mvp;\
     void main(void){\
        texcoord = inputtexcoord;\
        gl_Position = mvp * position;}",

    "varying highp vec2 texcoord;\
     uniform int noantialias;\
     uniform mediump float opacity;\
     uniform mediump vec4 basecolor;\
     uniform sampler2D basetexture;\
     void main(void){\
        mediump vec4 texlookup = texture2D(basetexture, texcoord);\
        gl_FragColor = basecolor;\
        if(noantialias == 0)\
          gl_FragColor.a = opacity * texlookup.a;\
        else\
         if (0.0 == texlookup.a)\
           gl_FragColor.a = 0.0;\
         else\
         gl_FragColor.a = opacity;}"};
static GLfloat projection[4][4] = {{2.0f/800.0f,0,0,0},{0,2.0f/480.0f,0,0},{0,0,0,0},{-1.0f,-1.0f,0,1.0f}};

static SVGBoolean   local_load_binary_shader(const GLuint *vertex_shader,
		                             const GLuint *fragment_shader);
		                             
#endif

static SVGError local_dumpTexToPPM( SVGFontFreeTypeRenderer* p_renderer,
								SVGInt8* data,
		                        SVGUint32 texW,
		                        SVGUint32 texH,
		                        SVGInt8 *file_name);

#ifdef HAS_OPENVG
static SVGError local_dumpTexVG(SVGFontFreeTypeRenderer* p_renderer,
						    SVGUint32 texW,
						    SVGUint32 TexH,
						    SVGInt8 *dest_buffer,
						    SVGInt32 is_filled);
#endif

#ifdef HAS_OPENGLES1
static GLfloat projection[4][4] = {{2.0f/800.0f,0,0,0},{0,2.0f/480.0f,0,0},{0,0,0,0},{-1.0f,-1.0f,0,1.0f}};
#endif

#ifdef HAS_OPENGLES

static PFNGLMULTIDRAWARRAYSEXTPROC glMultiDrawArraysEXT = NULL;

static void local_createFBO( GLuint *FBO,
		                     GLuint *depthbuffer,
		                     GLuint *colorbuffer,
		                     SVGUint32 w,
		                     SVGUint32 h);

static SVGError local_dumpTexGLES2( SVGFontFreeTypeRenderer* p_renderer,
		                        SVGUint32 texW,
		                        SVGUint32 texH,
		                        SVGInt8 *dest_buffer,
		                        SVGInt32 is_filled);
#endif

#ifndef HAS_OPENVG
#ifndef HAS_OPENGLES
#error ("defining at least one drawing API is mandatory!")
#endif
#endif
#include <stdlib.h>
#include <string.h>
#include <math.h>

#define min(a, b) (((a) < (b))? (a):(b))
#define max(a, b) (((a) > (b))? (a):(b))

#define min3(a, b, c) (((a) < (b))? min((a), (c)):min((b), (c)))
#define max3(a, b, c) (((a) > (b))? max((a), (c)):max((b), (c)))

#define min4(a, b, c, d) (((a) < (b))? min3((a), (c), (d)):min3((b), (c), (d)))
#define max4(a, b, c, d) (((a) > (b))? max3((a), (c), (d)):max3((b), (c), (d)))

grl_fontmodule_fp  grl_ft;
static SVGUint32    max_retries         = 5;/* matches 5 times 20%(default cache free) */
/*gap between glyphs in texture*/
//static SVGInt32     texture_gap         = 1;
static SVGInt32     texture_Ygap        = 2;

static SVGUint32    max_cache_entries   = 0;

static SVGUint32	cache_cleanup_ratio = 0;

static SVGUint8 	prn_buf[200] = {0};

static AUTOLANG_DB gs_autolangdb = { NULL,NULL,SVG_FALSE,{0}};

static SVGError local_cleanup_cache(SVGFontFreeTypeRenderer*   p_renderer);

static void local_check_parallel_use( SVGInt32 thread_count );

static void local_create_wallpapers( SVGFontContext* ctx , SVGUint32 fontSize );

static void local_free_wallpapers( SVGFontContext* ctx );

static void local_del_strk_glyphs( SVGFontContext* ctx );

static SVGBoolean   local_set_font( SVGFontContext* ctx,
                                   SVGFont* font,
                                   SVGEncoding charMap,
                                   SVGFontFreeTypeRenderer* p_renderer);

static void         local_calc_kerning_offset( SVGFontContext* ctx,
                                              get_char_info_struct* char_info);

static SVGError local_get_bitmap( SVGFontContext* p_ctx,
                           get_char_info_struct* char_info,
                           SVGUint32 loop,
                           SVGBoolean cache_bitmap,
                           SVGBoolean *texture_update,
                           SVGUint32	*theglyph);

static SVGUint8*    local_decode_char( get_char_info_struct* char_info);

static void         local_setFontMask( const SVGFontContext* ctx,
                                      get_char_info_struct *char_info);
                                      
static void         local_calc_maximum_extension( SVGFontContext* ctx,
                                                 SVGBBox  *textBBox,
                                                 SVGPoint *position,
                                                 SVGBoolean first_or_last_char);

static SVGUint32    local_drawFontWithCursorInfo( SVGFontContext* ctx,
                                                 SVGPoint* position,
                                                 SVGFloat* angle,
                                                 SVGUint8* text,
                                                 SVGUint32 textLength,
                                                 SVGPoint* cursorPosArray,
                                                 SVGBoolean draw_ext_glyphs,
                                                 SVGBBox* textBBox);

static SVGBoolean   local_create_shared_resources(SVGFontFreeTypeRenderer* p_renderer,
                                                 SVGUint32 textLength);

static void         local_destroy_shared_resources(SVGFontFreeTypeRenderer* p_renderer);

SVGError local_add_wallpaper_glyph_internal(SVGFontFreeTypeRenderer*   p_renderer,
                                             grl_internal_cache_struct* glyph,
                                             SVGUint8*                  buffer);

static SVGError     local_add_wallpaper_glyph(SVGFontFreeTypeRenderer*   p_renderer,
                                             grl_internal_cache_struct* glyph,
                                             SVGUint8*                  buffer);

static void         local_del_wallpaper_glyph(SVGFontFreeTypeRenderer*   p_renderer,
                                             grl_internal_cache_struct* glyph);

static BOOL			local_hash_cmp_fn(UTIL_hash_elem_t* e1, UTIL_hash_elem_t* e2);

static void 		reset_in_use_glyph_flag(SVGFontFreeTypeRenderer* p_renderer);

static SVGBoolean local_shpeng_find_scripttag ( get_char_info_struct*    p_charinfo,
                                                                  SVGUint32* p_scripttag,
                                                                 SVGUint32* p_rendrdirectag);

static SVGError local_shpeng_set_langscript_rendrdirection(SVGFontFreeTypeRenderer *p_renderer,
                                                                                           SVGLangScript   langscript,
                                                                                           SVGRenderDirection  rendr_direction);

static SVGBoolean local_shpeng_get_autolang_config(void);

#ifdef HAS_OPENVG
static void         local_render_vg_glyph(const SVGFontContext* ctx);
#endif

#ifdef HAS_OPENGLES
static void 		local_store_gl_bitmap(const SVGFontContext* ctx,
				                           get_char_info_struct* char_info,
				                           SVGUint32 l_index);

static void          local_render_gl_glyph(const SVGFontContext* ctx);
#endif
    					           
static void			local_clear_cache(SVGFontFreeTypeRenderer* p_renderer, SVGBoolean filled);
								   
#ifdef HAS_OPENGLES2
static void local_matrixTranslate(SVGFloat pMatrix[4][4], SVGFloat fX, SVGFloat fY);
static void local_matrixScale(SVGFloat pMatrix[4][4], SVGFloat fX, SVGFloat fY);
static void local_matrixRotate(SVGFloat pMatrix[4][4], SVGFloat angle);
static void local_matrixMutiply(SVGFloat psRes[4][4], SVGFloat psSrcA[4][4], SVGFloat psSrcB[4][4]);
static void local_matrixIdentity(SVGFloat pMatrix[4][4]);


static void local_matrixIdentity(SVGFloat pMatrix[4][4])
{
	pMatrix[0][0] = 1.0f;
	pMatrix[0][1] = 0.0f;
	pMatrix[0][2] = 0.0f;
	pMatrix[0][3] = 0.0f;
	pMatrix[1][0] = 0.0f;
	pMatrix[1][1] = 1.0f;
	pMatrix[1][2] = 0.0f;
	pMatrix[1][3] = 0.0f;
	pMatrix[2][0] = 0.0f;
	pMatrix[2][1] = 0.0f;
	pMatrix[2][2] = 1.0f;
	pMatrix[2][3] = 0.0f;
	pMatrix[3][0] = 0.0f;
	pMatrix[3][1] = 0.0f;
	pMatrix[3][2] = 0.0f;
	pMatrix[3][3] = 1.0f;
}

static void local_matrixMutiply(SVGFloat psRes[4][4], SVGFloat psSrcA[4][4], SVGFloat psSrcB[4][4])
{
	SVGFloat fB00, fB01, fB02, fB03;
	SVGFloat fB10, fB11, fB12, fB13;
	SVGFloat fB20, fB21, fB22, fB23;
	SVGFloat fB30, fB31, fB32, fB33;
	SVGUint32 i;

	fB00 = psSrcB[0][0]; fB01 = psSrcB[0][1]; fB02 = psSrcB[0][2]; fB03 = psSrcB[0][3];
	fB10 = psSrcB[1][0]; fB11 = psSrcB[1][1]; fB12 = psSrcB[1][2]; fB13 = psSrcB[1][3];
	fB20 = psSrcB[2][0]; fB21 = psSrcB[2][1]; fB22 = psSrcB[2][2]; fB23 = psSrcB[2][3];
	fB30 = psSrcB[3][0]; fB31 = psSrcB[3][1]; fB32 = psSrcB[3][2]; fB33 = psSrcB[3][3];

	for (i = 0; i < 4; i++) 
	{
		psRes[i][0] = psSrcA[i][0]*fB00 + psSrcA[i][1]*fB10	+ psSrcA[i][2]*fB20 + psSrcA[i][3]*fB30;
		psRes[i][1] = psSrcA[i][0]*fB01 + psSrcA[i][1]*fB11	+ psSrcA[i][2]*fB21 + psSrcA[i][3]*fB31;
		psRes[i][2] = psSrcA[i][0]*fB02 + psSrcA[i][1]*fB12	+ psSrcA[i][2]*fB22 + psSrcA[i][3]*fB32;
		psRes[i][3] = psSrcA[i][0]*fB03 + psSrcA[i][1]*fB13	+ psSrcA[i][2]*fB23 + psSrcA[i][3]*fB33;
	}
}

static void local_matrixScale(SVGFloat pMatrix[4][4], SVGFloat fX, SVGFloat fY)
{
	pMatrix[0][0] *= fX;
	pMatrix[0][1] *= fX;
	pMatrix[0][2] *= fX;
	pMatrix[0][3] *= fX;
	pMatrix[1][0] *= fY;
	pMatrix[1][1] *= fY;
	pMatrix[1][2] *= fY;
	pMatrix[1][3] *= fY;
}

static void local_matrixTranslate(SVGFloat pMatrix[4][4], SVGFloat fX, SVGFloat fY)
{
	pMatrix[3][0] += fX * pMatrix[0][0] + fY * pMatrix[1][0];
	pMatrix[3][1] += fX * pMatrix[0][1] + fY * pMatrix[1][1];
	pMatrix[3][2] += fX * pMatrix[0][2] + fY * pMatrix[1][2];
	pMatrix[3][3] += fX * pMatrix[0][3] + fY * pMatrix[1][3];
}

static void local_matrixRotate(SVGFloat pMatrix[4][4], SVGFloat angle)
{
    SVGFloat cosin_angle    = SVG_COSINE_LUT[(SVGUint32)angle];
    SVGFloat sin_angle      = SVG_SINE_LUT[(SVGUint32)angle];
	SVGFloat afMatrix[4][4] = {{1,0,0,0},{0,1,0,0},{0,0,1,0},{0,0,0,1}};

	afMatrix[0][0] = cosin_angle;
	afMatrix[1][1] = cosin_angle;
	afMatrix[1][0] = -sin_angle;
	afMatrix[0][1] = sin_angle;

	local_matrixMutiply(pMatrix, afMatrix, pMatrix);
}
#endif

static void local_check_parallel_use( SVGInt32 thread_count )
{
	if (thread_count > 1)
	{
		FILE *err_mem;
		SVGInt32 pid = getpid();
	    SVG_FNT_F("NOT ALLOWED PARALLEL ACCESS, \
	    		one svgfont context is used from several threads,pid %d",pid);
		err_mem = fopen("/dev/errmem","wa");
		if (err_mem)
		{
			fprintf(err_mem,"SVGFONT ERROR: NOT ALLOWED PARALLEL ACCESS, one font context is used from several threads,pid: %d \n",pid);
			fflush(err_mem);
			fclose(err_mem);
		}
	}
}

void grlft_get_font_suffix(S8 **suffix_list,U32 *list_size)
{
    *suffix_list = (S8*)FT_SUPPORTED_SUFFIXES;
    *list_size   = FT_NR_OF_SUP_SUFFIXES;
}

void grlft_fill_font_struct(SVGUint32 *font_filenames, SVGUint32 num_files, SVGFont **font_data, SVGUint32 *max_fonts)
{
    FT_Library              library     = NULL;
    FT_Face                 aface       = NULL;
    FT_CharMap              charmap     = NULL;
    SVGUint8                *address    = NULL;
    SVGUint32               len         = 0;
    SVGUint16               loop        = 0;
    FT_Bitmap_Size          *size       = NULL;
    SVGBoolean              plain       = SVG_TRUE;
    SVGBoolean              isSupported = SVG_FALSE;
	SVGFont					*font		= NULL;
	SVGUint32				i 			= 0;
	SVGInt8					*font_filename 	= NULL;
	SVGUint32				faceIndex 	= 0;
	SVGUint32				*num_faces_per_file= (SVGUint32*)GRL_malloc_1D_resource(sizeof(SVGInt32) * num_files);
	SVGInt32				act_font	= 0;
	SVGBoolean              do_load     = SVG_TRUE;

	if( (NULL != num_faces_per_file) && 
		(NULL != font_data) && 
		(NULL != max_fonts) )
	{
		/* create temporary face */
		if (FT_Init_FreeType(&library) != 0 )
		{
		    SVG_FNT_E("CREATE_FREETYPE_LIB_FAILED IN GRLXX_FILL_FONT_STRUCT");
		}
		else
		{
			for(i = 0; i < num_files; i++)
			{
			/*find out how many face are in each file (.ttc can contain more then one)*/
				font_filename = (SVGInt8*)(uintptr_t)font_filenames[i];
				aface = NULL;
				
				if( NULL != font_filename)
				{
				    do_load = SVG_TRUE;
                    if((NULL != strstr((const SVGChar*)font_filename, ".ccc")) ||
                       (NULL != strstr((const SVGChar*)font_filename, ".ltt")))
                    {
                        if(0 == FT_Get_Module( library, "itype drv" ))
                        {
                            do_load = SVG_FALSE;
                        }
                    }

                    if(SVG_TRUE == do_load)
                    {
                        if( NULL == strstr((const SVGChar*)font_filename, (const SVGChar*)FF_ROOT))
                        {
                            //find out how many faces (ttc files can countain more then one)
                            if (FT_New_Face(library, (const SVGChar*)font_filename, -1, &aface) != 0)
                            {
                    		    SVG_FNT_E("OPEN_FILE_FONT_ERROR IN GRLXX_FILL_FONT_STRUCT");
                            }
                        }
                        else
                        {
                            /* fixed flash memory font */
                            /* first split the "filename" into address and len ...*/
                            memcpy(&address, (&font_filename[FF_ROOT_LEN]), FF_INTEGER_LEN);
                            memcpy(&len, (&font_filename[FF_ROOT_LEN + FF_INTEGER_LEN + 1]), FF_INTEGER_LEN);
                            if (FT_New_Memory_Face(library,  address, (SVGInt64)len, -1, &aface) != 0)
                            {
                    		    SVG_FNT_E("OPEN_FILE_FONT_ERROR IN GRLXX_FILL_FONT_STRUCT");
                            }
                        }
                    }
					if(aface != NULL)
					{
						num_faces_per_file[i] = (SVGUint32)aface->num_faces ;
						*max_fonts += (SVGUint32)aface->num_faces;
					}
					else
					{
						num_faces_per_file[i] = 0;
					}
				}
			}
			/*allocate memory for all faces and load them*/
			*font_data = (SVGFont*)GRL_malloc_1D_resource(sizeof(SVGFont) * (*max_fonts));
			if( NULL != *font_data)
			{
				for(i = 0; i < num_files; i++)
				{

					/* updating the plain value for every font file face */
					plain       = SVG_TRUE;

					font_filename = (SVGInt8*)(uintptr_t)font_filenames[i];
					for(faceIndex = 0; faceIndex < num_faces_per_file[i]; faceIndex++)
					{

						font = &font_data[ 0 ][act_font++];
						aface = NULL;
						isSupported = SVG_FALSE;
						
						if(NULL == strstr((const SVGChar*)font_filename, (const SVGChar*)FF_ROOT))
						{	
							/* font file from filesystem */
							if (FT_New_Face(library, (const SVGChar*)font_filename, (SVGInt64)faceIndex, &aface) != 0)
							{
                    		    SVG_FNT_E("OPEN_FILE_FONT_ERROR IN GRLXX_FILL_FONT_STRUCT");
							}
							strncpy((SVGChar*)font->filename, (const SVGChar*)font_filename, SVG_MAX_PATH );
						}
						else
						{
							/* fixed flash memory font */
							/* first split the "filename" into address and len ...*/
							memcpy(&address, (&font_filename[FF_ROOT_LEN]), FF_INTEGER_LEN);
							memcpy(&len, (&font_filename[FF_ROOT_LEN + FF_INTEGER_LEN + 1]), FF_INTEGER_LEN);
							memcpy(font->filename, font_filename, FF_NAME_LENGTH - 1);
							/* ... and then open the font */
							if(FT_New_Memory_Face(library, address, (SVGInt64)len, (SVGInt64)faceIndex, &aface) != 0)
							{
                    		    SVG_FNT_E("OPEN_MEM_FONT_ERROR IN GRLXX_FILL_FONT_STRUCT");
							}
						}
						font->faceIndex = faceIndex;
						font->attributes = 0;
						/* when something went wrong just return */
						if(NULL == aface)
						{
							font->fontName[0] = '\0';
						}
						else
						{
							if (aface->num_charmaps > 1)
							{
								/* do limitation to font which supports UNICODE/ASCII */
								for(loop = 0; loop < aface->num_charmaps; loop++)
								{
									charmap = aface->charmaps[loop];
									if (((charmap->platform_id == 0) &&
										(charmap->encoding == FT_ENCODING_NONE)) ||
										((charmap->platform_id == 3) &&
										(charmap->encoding == FT_ENCODING_UNICODE)))
									{
										isSupported = SVG_TRUE;
									}
								}
							}
							else
							{
								/* workaround for bitmaps fonts (only one charmap) */
								if (aface->num_charmaps == 1)
								{
									charmap = aface->charmaps[0];
									if (((charmap->platform_id == 0) &&
										(charmap->encoding == FT_ENCODING_NONE)) ||
										((charmap->platform_id == 3) &&
										(charmap->encoding == FT_ENCODING_UNICODE)))
									{
										isSupported = SVG_TRUE;
									}
								}
							}

							if (isSupported == SVG_TRUE)
							{

								/* copy the fontname to the font struct */
								strncpy((SVGChar*)font->fontName, (const SVGChar*)aface->family_name, 64);

								if (FT_FACE_FLAG_SCALABLE ==
								   (((SVGUint32)aface->face_flags) & FT_FACE_FLAG_SCALABLE))

								{
									font->attributes |= SVG_FONT_SCALABLE;
								}

								if (FT_FACE_FLAG_VERTICAL ==
								   (((SVGUint32)aface->face_flags) & FT_FACE_FLAG_VERTICAL))
								{
									font->attributes |= SVG_FONT_VERTICAL;
								}

								if (FT_FACE_FLAG_KERNING ==
									(((SVGUint32)aface->face_flags) & FT_FACE_FLAG_KERNING))
								{
									font->attributes |= SVG_FONT_KERNING;
								}

								if (FT_FACE_FLAG_FIXED_WIDTH ==
								(((SVGUint32)aface->face_flags) & FT_FACE_FLAG_FIXED_WIDTH))
								{
									font->attributes |= SVG_FONT_FIXED;
								}
								else
								{
									font->attributes |= SVG_FONT_PROP;
								}

								/* has the font italic chars*/
								if (FT_STYLE_FLAG_ITALIC ==
								   (((SVGUint32)aface->style_flags) & FT_STYLE_FLAG_ITALIC))
								{
									font->attributes |= SVG_FONT_ITALIC;
									plain = SVG_FALSE;
								}

								/* has the font bold chars*/
								if (FT_STYLE_FLAG_BOLD ==
									(((SVGUint32)aface->style_flags) & FT_STYLE_FLAG_BOLD))
								{
									font->attributes |= SVG_FONT_BOLD;
									plain = SVG_FALSE;
								}

								/* has the font neither bold nor italic chars */
								if (SVG_TRUE == plain)
								{
									font->attributes |= SVG_FONT_PLAIN;
								}


								if(FT_FACE_FLAG_FIXED_SIZES ==
								(((SVGUint32)aface->face_flags) & FT_FACE_FLAG_FIXED_SIZES))
								{
									font->lengthSizes = (SVGUint8) aface->num_fixed_sizes;
									font->allowedSizes[0] = '\0';
									size = aface->available_sizes;

									for(loop = 0;
										(loop < aface->num_fixed_sizes) && (loop < 32);
										loop++)
									{
										font->allowedSizes[loop] = (SVGUint8)(size->x_ppem >> 6);
										size = &size[1];
									}
								}
								else
								{
									font->lengthSizes     = 0;
									font->allowedSizes[0] = '\0';
								}

								if(aface->num_charmaps > 0)
								{
									font->lengthMaps     = (SVGUint8) aface->num_charmaps;
									/* tbd dlangner copy available charmaps */
									font->charMaps[0]    = '\0';
								}
								else
								{
									font->lengthMaps     = 0;
									font->charMaps[0]    = '\0';
								}
								font->faceType       = 0;
								if (FT_Done_Face(aface) != 0)
								{
									 SVG_FNT_E("DESTROY_FREETYPE_FACE_FAILED IN "
											 "IN GRLXX_FILL_FONT_STRUCT");
								}
							}
							else
							{
								act_font--;
								if(*max_fonts != 0)
									*max_fonts = *max_fonts -1 ;
							}
						}
					}
				}
			}
		}
		if ( FT_Done_FreeType(library) != 0 )
		{
		    SVG_FNT_E("DESTROY_FREETYPE_LIB_FAILED "
		    		  "IN GRLXX_FILL_FONT_STRUCT");
		}
	}
	if( NULL != num_faces_per_file)
		GRL_free_1D_resource( num_faces_per_file );
}


void grlft_init_font_plugin(SVGFontModule *module)
{
    SVGInt32 a_length = 0;
    grl_ft.GetFontError           = grlftGetFontError;
    grl_ft.CreateFontContext      = grlftCreateFontContext;
    grl_ft.DestroyFontContext     = grlftDestroyFontContext;
    grl_ft.SetFontSize            = grlftSetFontSize;
    grl_ft.LoadFont               = grlftLoadFont;
    grl_ft.EnableFontSettings     = grlftEnableFontSettings;
    grl_ft.DisableFontSettings    = grlftDisableFontSettings;
    grl_ft.IsFontSettingEnabled   = grlftIsFontSettingEnabled;
    grl_ft.GetFontBitmap          = grlftGetFontBitmap;
    grl_ft.FreeFontBitmapMem      = grlftFreeFontBitmapMem;
    grl_ft.DrawFontWithCursorInfo = grlftDrawFontWithCursorInfo;
    grl_ft.GetFontBBox            = grlftGetFontBBox;
    grl_ft.get_font_suffix        = grlft_get_font_suffix;
    grl_ft.fill_font_structs      = grlft_fill_font_struct;
    grl_ft.GetFontMaxChars        = grlftGetFontMaxChars;
    grl_ft.GetFontInformation     = grlftGetFontInformation;
    grl_ft.DrawFontStringExt      = grlftDrawFontStringExt;
    grl_ft.GetFontBitmapExt       = grlftGetFontBitmapExt;
    grl_ft.DumpFontWallpaper	  = grlftDumpFontWallpaper;
    grl_ft.SetCachableGlyphCount  = grlftSetCachableGlyphCount;
    module->id = SVG_PLUGIN_FREETYPE;
    strcpy((SVGChar*)module->name, (const SVGChar*)"FreeType");
    module->p_grl_fp_table  = &grl_ft;

    a_length = GRL_get_int_config((SVGChar*)"SVGFNTCACHECNT", (SVGInt32*)&max_cache_entries, 0);
    if( a_length > 0)
    {
        GRL_get_int_config((SVGChar*)"SVGFNTCACHECNT", (SVGInt32*)&max_cache_entries, (SVGUint8)a_length);
    }
    else
    {
        max_cache_entries = MAX_CACHE_ENTRIES;
    }    

    a_length = GRL_get_int_config((SVGChar*)"SVGFNTCACHECLEANRATIO", (SVGInt32*)&cache_cleanup_ratio, 0);
	if( a_length > 0)
	{
		GRL_get_int_config((SVGChar*)"SVGFNTCACHECLEANRATIO", (SVGInt32*)&cache_cleanup_ratio, (SVGUint8)a_length);
	}
	else
	{
		cache_cleanup_ratio = CACHE_CLEANUP_RATIO;
	}
	max_retries = 100 / cache_cleanup_ratio;
}

void grlft_close_font_plugin(void)
{
	;    
}

static SVGError local_dumpTexToPPM( SVGFontFreeTypeRenderer*    p_renderer, SVGInt8* data, SVGUint32 texW, SVGUint32 texH, SVGInt8 *file_name)
{
    SVGError error     = GRL_NO_ERROR;
	FILE    *fileDump  = NULL;
	SVGUint32 i         = 0;
	SVGUint32 j         = 0;
	SVGUint32 size      = 0;

	fileDump = fopen((char*)file_name, "wb");

	if (fileDump)
	{
		if ( data)
		{
#ifdef HAS_OPENGLES
			if(SVG_USE_OPENGLES_1 == p_renderer->drawingAPI)
			{
				fprintf(fileDump, "P6\n%d %d\n255\n", texW, texH);
				size = texW * texH *3;
				for (i = 0; i < size; i += 3)
				{
					data[i]     = data[j+3]; 		/* red */
					data[i + 1] = data[j + 3]; 	/* green */
					data[i + 2] = data[j + 3]; 	/* blue */
					j += 4;
				}
			}
#endif
#ifdef HAS_OPENVG
			if(SVG_USE_OPENVG == p_renderer->drawingAPI)
			{
				fprintf(fileDump, "P5\n%d %d\n255\n", texW, texH);
				size = texW * texH;
            }
#endif
			(void) fwrite(data, 1, size, fileDump);
		}
		else
		{
			error =	SVG_POINTER_NULL;
		}
		fclose(fileDump);
	}
	else
	{
		error = OPEN_DUMP_FILE_ERROR;
	}

	return error;
}

#ifdef HAS_OPENVG
SVGError local_dumpTexVG(SVGFontFreeTypeRenderer*    p_renderer,SVGUint32 texW,SVGUint32 texH, SVGInt8 *dest_buffer, SVGInt32 is_filled)
{
    SVGError 	error           = GRL_NO_ERROR;
	SVGInt8 	*dump           = NULL;
	SVGUint32 	i               = 0;
	SVGUint32 	w               = texW;
	SVGUint32 	h               = texH;
	SVGUint32	size            = texW * texH ;
	SVGUint32	stride          = w ;

	dump = (SVGInt8*) GRL_malloc_1D_resource(sizeof(SVGInt8) * size );

	if(dump != NULL)
	{
		if (is_filled == 0)
		{
			vgGetImageSubData(p_renderer->shared->vgWallpaperStrk,
							  dump,
							  (VGint)texW,
							  VG_A_8,
							  0,0,
							  (VGint)texW, (VGint)texH);
		}
		else
		{
			vgGetImageSubData(p_renderer->shared->vgWallpaperFill,
							  dump,
							  (VGint)texW,
							  VG_A_8,
							  0,0,
							  (VGint)texW, (VGint)texH);
		}

		memset( dest_buffer, 0x0, (SVGUint32) size );
		for( i = 0; i < h ; i++)
		{
			memcpy( &dest_buffer[ ((h - 1) - i) * stride] ,
					&dump[ i * stride ],
					 stride );
		}

		GRL_free_1D_resource(dump);
	}
	else
	{
		error = OPEN_DUMP_FILE_ERROR;
	}

	return error;
}
#endif

#ifdef HAS_OPENGLES2
static void local_createFBO( GLuint *FBO, GLuint *depthbuffer, GLuint *colorbuffer, SVGUint32 w, SVGUint32 h)
{
	glGenFramebuffers(1, FBO);
	glBindFramebuffer(GL_FRAMEBUFFER, *FBO);

	glGenRenderbuffers(1, depthbuffer);
	glBindRenderbuffer(GL_RENDERBUFFER, *depthbuffer);
	glRenderbufferStorage(GL_RENDERBUFFER, GL_DEPTH_COMPONENT16, (GLsizei)w, (GLsizei)h);
	glFramebufferRenderbuffer(GL_FRAMEBUFFER, GL_DEPTH_ATTACHMENT,GL_RENDERBUFFER, *depthbuffer);
	
	glGenRenderbuffers(1, colorbuffer);
	glBindRenderbuffer(GL_RENDERBUFFER, *colorbuffer);
	glRenderbufferStorage(GL_RENDERBUFFER, GL_RGBA4,(GLsizei) w, (GLsizei)h);
	glFramebufferRenderbuffer(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0,GL_RENDERBUFFER, *colorbuffer);
}

static SVGError local_dumpTexGLES2( SVGFontFreeTypeRenderer*    p_renderer,SVGUint32 texW, SVGUint32 texH, SVGInt8 *dest_buffer, SVGInt32 is_filled)
{
    SVGError    error           = GRL_NO_ERROR;
	SVGInt8 	*dump           = NULL;
	SVGUint32 	i     	        = 0;
	SVGUint32 	w               = texW;
	SVGUint32 	h               = texH;
	SVGUint32	size            = texW * texH * 4;
	SVGUint32	stride          = w * 4;
	GLfloat 	vertx[8];
	GLfloat 	texCoords[8];
	GLfloat		drawCol[]       = {1.0f, 1.0f, 1.0f, 1.0f};
	GLfloat		clearCol[4];
	

	vertx[0] = 0.0f;
	vertx[1] = 0.0f;
	vertx[3] = 0.0f;
	vertx[6] = 0.0f;
	vertx[2] = texW;
	vertx[4] = vertx[2];
	vertx[5] = texH;
	vertx[7] = vertx[5];

	/* source vertex */
	texCoords[0] = 0;
	texCoords[1] = 0;
	texCoords[2] = texCoords[0] + 1;
	texCoords[3] = texCoords[1];
	texCoords[4] = texCoords[2];
	texCoords[5] = texCoords[3] + 1;
	texCoords[6] = texCoords[0];
	texCoords[7] = texCoords[5];

	w = h = texW;


	if (is_filled == 0)
	{
		glBindTexture(GL_TEXTURE_2D, p_renderer->shared->glWallpaperStrk);
	}
	else
	{
		glBindTexture(GL_TEXTURE_2D, p_renderer->shared->glWallpaperFill);
	}

	glUseProgram(p_renderer->shared->programHandle);
	glEnableVertexAttribArray (p_renderer->shared->vertexlocation);
	glEnableVertexAttribArray (p_renderer->shared->texturelocation);
	glUniform1f(p_renderer->shared->opacitylocation, 1.0f);
	glUniform1i(p_renderer->shared->noaalocation, 0);
	projection[0][0] = 2.0f / texW;
	projection[1][1] = 2.0f / texH;
	local_matrixIdentity(p_renderer->shared->modelview);
	local_matrixMutiply(p_renderer->shared->mvp, p_renderer->shared->modelview, projection);
	glUniformMatrix4fv(p_renderer->shared->mvplocation, 1, GL_FALSE, &p_renderer->shared->mvp[0][0]);
	glUniform4fv(p_renderer->shared->colorlocation, 1, &drawCol[0]);
	
	glDisable(GL_BLEND);
	//glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	
	glGetFloatv(GL_COLOR_CLEAR_VALUE, clearCol);
	glClearColor(0.0f, 0.0f, 0.0f, 0.0f);
	glClear(GL_COLOR_BUFFER_BIT );
		
	glVertexAttribPointer(p_renderer->shared->texturelocation, 2, GL_FLOAT, 0, 0, texCoords);
	glVertexAttribPointer(p_renderer->shared->vertexlocation, 2, GL_FLOAT, 0, 0, vertx);

	glDrawArrays(GL_TRIANGLE_FAN, 0, 4);
	
	dump = (SVGInt8*) GRL_malloc_1D_resource(sizeof(SVGInt8) * size );

	if(dump != NULL)
	{
		glReadPixels(0, 0, (GLsizei)w,(GLsizei) h, GL_RGBA, GL_UNSIGNED_BYTE, dump);

		memset( dest_buffer, 0x0, (SVGUint32) size );
		for( i = 0; i < h ; i++)
		{
			memcpy( &dest_buffer[ ((h - 1) - i) * stride] ,
					&dump[ i * stride ],
					stride );
		}
		GRL_free_1D_resource(dump);
	}
	else
	{
		error = OPEN_DUMP_FILE_ERROR;
	}
/*restore clear color values*/
	projection[0][0] = 2.0f / p_renderer->screen_width;
	projection[1][1] = 2.0f / p_renderer->screen_height;
	glEnable(GL_BLEND);
	glClearColor(clearCol[0],clearCol[1],clearCol[2],clearCol[3]);

	return error;
}
#endif

void grlftDumpFontWallpaper( SVGFontContext* ctx, 
							 SVGUint32 		 settings, 
							 SVGInt8* 		 fileName)
{
    SVGError    error        = GRL_NO_ERROR;
    SVGUint32   error_api    = 0;
	SVGInt32	i            = 0;
	SVGInt32	start        = 1;
	SVGInt32	end          = 2;
	SVGUint32	texW         = 0;
	SVGUint32	texH         = 0;
	SVGInt8 	out_buf[256] = "";
	SVGInt8		final_dump_name[256];
	SVGInt8		*dest_buffer = NULL;
#ifdef HAS_OPENGLES
	SVGInt32	current_program    = 0;
	GLuint		FBO          = 0;
	GLuint		depthbuffer  = 0;
	GLuint		colorbuffer  = 0;
#endif	
	SVGFontFreeTypeRenderer*    p_renderer	= (SVGFontFreeTypeRenderer*) ctx->p_fontRendererContext;

	if (fileName == NULL)
	{
#ifdef __linux__
		strcat((char*)out_buf, (char*)"/tmp/\0");
#else
		strcat((char*)out_buf, (char*)"/host/\0");
#endif
		strcat((char*)out_buf, "FontWallpaper");
	}
	else
	{
		strcat((char*)out_buf, (char*)fileName);
	}
	
	if( (settings & SVG_FONT_OUTLINE) != 0)
	{
		start = 0;
		end = 1;
	}
	if( (settings & SVG_FONT_2COLOR) != 0)
	{
		start = 0;
		end = 2;
	}	
#ifdef HAS_OPENVG
	if(SVG_USE_OPENVG == p_renderer->drawingAPI)
	{
		error_api = vgGetError();
		texW = p_renderer->shared->vgWallpaperWidth;
		texH = p_renderer->shared->vgWallpaperHeight;
	}
#endif
#ifdef HAS_OPENGLES
	if(SVG_USE_OPENGLES_1 == p_renderer->drawingAPI)
	{     
		glGetIntegerv(GL_CURRENT_PROGRAM, &current_program);

		error_api = glGetError();
		texW = p_renderer->shared->glWallpaperWidth;
		texH = p_renderer->shared->glWallpaperHeight; 
		glViewport(0, 0, (GLsizei)texW,(GLsizei) texH);

#ifdef HAS_OPENGLES2
		local_createFBO( &FBO , &depthbuffer, &colorbuffer, texW, texH);
#else
		//not available with gles1
#endif
	}
#endif
	dest_buffer = (SVGInt8*) GRL_malloc_1D_resource(sizeof(SVGInt8)*texW*texH*4);

	if (dest_buffer != NULL)
	{
		for(i = start; i < end ; i++)
		{
			strcpy( (char*)final_dump_name, (char*)out_buf);
			if (i == 0)
			{
				strcat((char*)final_dump_name, "_STROKED");
			}
			else
			{
				strcat((char*)final_dump_name, "_FILLED");
			}
			strcat((char*)final_dump_name, ".ppm");
#ifdef HAS_OPENGLES
			if(SVG_USE_OPENGLES_1 == p_renderer->drawingAPI)
			{
#ifdef HAS_OPENGLES2
				error = local_dumpTexGLES2( p_renderer, texW, texH, dest_buffer, i);
#else
			//not available with gles1
#endif
			}
#endif
#ifdef HAS_OPENVG
			if (SVG_USE_OPENVG == p_renderer->drawingAPI)
			{
				error = local_dumpTexVG( p_renderer, texW, texH, dest_buffer, i );
			}
#endif
			error = local_dumpTexToPPM(p_renderer,dest_buffer, texH, texW, final_dump_name);

			if (error != GRL_NO_ERROR)
			{
				break;
			}
		}
	}
	else
	{
		error = SVG_OUT_OF_MEMORY;
	}

#ifdef HAS_OPENGLES
	if(SVG_USE_OPENGLES_1 == p_renderer->drawingAPI)
	{
#ifdef HAS_OPENGLES2
		glBindFramebuffer(GL_FRAMEBUFFER, 0);
		glDeleteFramebuffers(1, &FBO);
		glDeleteRenderbuffers(1, &depthbuffer);
		glDeleteRenderbuffers(1, &colorbuffer);
#else
		//gles1 clean up if necessary
#endif
		if ((ctx->settings & SVG_FONT_USE_CURRENT_PROGRAM) == 0)
		{
			glUseProgram((SVGUint32)current_program);
		}
	}

	if(NULL != ctx->shared)
    {
        GRL_wai_lock(p_renderer->shared->id);
    }
    __sync_add_and_fetch(&ctx->thread_count, 1);//atomic incrementation, used for parallel access detection
    local_check_parallel_use( ctx->thread_count );

	eglQuerySurface(eglGetCurrentDisplay(),
					eglGetCurrentSurface(EGL_DRAW),
					EGL_WIDTH,
					&p_renderer->screen_width);
	eglQuerySurface(eglGetCurrentDisplay(),
					eglGetCurrentSurface(EGL_DRAW),
					EGL_HEIGHT,
					&p_renderer->screen_height);

	glViewport(0, 0, p_renderer->screen_width,
					 p_renderer->screen_height);
	
	FBO = 0;
	depthbuffer = 0;
	colorbuffer = 0;

	local_check_parallel_use( ctx->thread_count );
	__sync_sub_and_fetch(&ctx->thread_count, 1);
	if(NULL != ctx->shared)
	{
		GRL_sig_lock(p_renderer->shared->id);
	}

#endif


	if( dest_buffer )
	{
		GRL_free_1D_resource( dest_buffer);
	}

	if (error != GRL_NO_ERROR)
	{
        grlftSetError(ctx, error);
        SVG_FNT_E("ERROR IN GRLXXDUMPFONTWALLPAPER");
	}
#ifdef HAS_OPENGLES
	if (SVG_USE_OPENGLES_1 == p_renderer->drawingAPI)
	{
		error_api = glGetError();
		if (GL_NO_ERROR != error_api)
		{
		    SVG_FNT_E("SVG_OPENGL_ERROR IN "
		    		"GRLXXDUMPFONTWALLPAPER %d",error_api);
		}
	}
#endif
#ifdef HAS_OPENVG
	if (SVG_USE_OPENVG == p_renderer->drawingAPI)
	{
		error_api = vgGetError();
		if (VG_NO_ERROR != error_api)
		{
		    SVG_FNT_E("SVG_OPENVG_ERROR IN GRLXXDRAWFONTWITHCURSORINFO %d",error_api);
		}
	}
#endif
}

void grlftSetError(SVGFontContext* ctx, SVGError error)
{
    if (SVG_NO_ERROR == ctx->error)
    {
        ctx->error = error;
    }
}

SVGError grlftGetFontError(SVGFontContext* ctx)
{
    SVGError    ret = SVG_POINTER_NULL;

    ret        = ctx->error;
    ctx->error = SVG_NO_ERROR;

    return ret;
}


SVGFontContext* grlftCreateFontContext( EGLContext ctx, SVGFontContext* shareContext )
{
    SVGFontContext          *newContext         = NULL;
    SVGFontFreeTypeRenderer *fontRenderer       = NULL;
    SVGError                error               = GRL_NO_ERROR;
#ifdef HAS_OPENGLES
    const GLubyte 			*pszGLExtensions;
#endif
#ifdef HAS_OPENGLES2
    SVGUint32               loop                = 0;
    SVGInt32                length              = 0;
    SVGInt32                status              = 0;
#endif
	
	UNREFED_FONT_PARAM(ctx)
    newContext = (SVGFontContext *)GRL_malloc_1D_resource(sizeof(SVGFontContext));
    if (NULL != newContext)
    {
		memset( newContext, 0 ,sizeof(SVGFontContext));
        fontRenderer = (SVGFontFreeTypeRenderer *)GRL_malloc_1D_resource(sizeof(SVGFontFreeTypeRenderer));
        if (NULL != fontRenderer)
        {
			memset(fontRenderer, 0 ,sizeof(SVGFontFreeTypeRenderer));

             fontRenderer->autolangdetect = local_shpeng_get_autolang_config();

            if(( NULL != shareContext) && (NULL != shareContext->p_fontRendererContext))
            {
                fontRenderer->shared = ((SVGFontFreeTypeRenderer *)(shareContext->p_fontRendererContext))->shared;
                GRL_wai_lock(fontRenderer->shared->id);
                /* increment shares count in shared context */
                shareContext->shares_count++;
                newContext->shared = shareContext;
                GRL_sig_lock(fontRenderer->shared->id);
            }
            else
            {
                fontRenderer->shared = GRL_malloc_1D_resource(sizeof(SVGFontShared));
                if(NULL != fontRenderer->shared)
                {
					memset(fontRenderer->shared, 0 ,sizeof(SVGFontShared));
                    if ( 0 != FT_Init_FreeType( &fontRenderer->shared->library ))
                    {
                        error = SVG_FREETYPE_ERROR;
            		    SVG_FNT_F("INITIALIZE_FREETYPE_FAILED IN "
            		    		     "GRLXXCREATEFONTCONTEXT ");
                    }
                    else
                    {
                        fontRenderer->shared->mem_guard1 = 0xdeadbeef;
                        fontRenderer->shared->mem_guard2 = 0xdeadbeef;
                        newContext->validFont                       = SVG_FALSE;
                        newContext->shared                          = NULL;
                        fontRenderer->texture                       = 0;
                        fontRenderer->shared->p_svgFont             = NULL;
                        fontRenderer->shared->late_glyphs_size      = 0xffffffff;
                        fontRenderer->shared->late_glyphs           = NULL;
                        fontRenderer->shared->cache                 = NULL;
                        fontRenderer->shared->cache_count_strk      = 0;
						fontRenderer->shared->cache_count_fill      = 0;
						fontRenderer->shared->cache_count      		= &fontRenderer->shared->cache_count_strk;
                        fontRenderer->shared->cache_current         = NULL;
                        fontRenderer->cachable_glyphs               = max_cache_entries;
                        fontRenderer->shared->hash_tbl_strk		= UTIL_cre_hsh(fontRenderer->cachable_glyphs, local_hash_cmp_fn );
                        fontRenderer->shared->hash_tbl_fill		= UTIL_cre_hsh(fontRenderer->cachable_glyphs, local_hash_cmp_fn );
                        if( fontRenderer->shared->hash_tbl_strk == NULL )
                        {
                        	error = SVG_FREETYPE_ERROR;
                		    SVG_FNT_F("SVG_OUT_OF_MEMORY IN \
                		    		     GRLXXCREATEFONTCONTEXT ");
                        }
						if( fontRenderer->shared->hash_tbl_fill == NULL )
                        {
                        	error = SVG_FREETYPE_ERROR;
                		    SVG_FNT_F("SVG_OUT_OF_MEMORY IN GRLXXCREATEFONTCONTEXT ");
                        }
#ifdef HAS_OPENVG
                        fontRenderer->shared->vgWallpaperStrk       = 0;
                        fontRenderer->shared->vgWallpaperFill       = 0;
#endif                        
#ifdef HAS_OPENGLES
                		fontRenderer->shared->glWallpaperStrk       = 0;
                        fontRenderer->shared->glWallpaperFill       = 0;
                        /*set the default unpack alignment for texture to 4*/
                        fontRenderer->glUnpackAlignment             = 4;
#endif                        
                        GRL_cre_lock(&fontRenderer->shared->id, NULL);
                        fontRenderer->stroker_availabe = SVG_TRUE;
                        if(0 != FT_Stroker_New(fontRenderer->shared->library, &fontRenderer->shared->stroker))
                        {
                             grlftSetError(newContext, STROKER_CREATION_FAILED);
                 		     SVG_FNT_F("STROKER_CREATION_FAILED IN \
                 		    		     GRLXXCREATEFONTCONTEXT ");
                            fontRenderer->stroker_availabe = SVG_FALSE;                                      
                        }

                        /* Init shaping engine object with invalid render direction and script for new font context.*/
                        if( SVG_TRUE == fontRenderer->autolangdetect )
                        {
                            if( SVG_NO_ERROR ==  grl_initShapeEng_Plugin(&(fontRenderer->shared->p_shape_eng),
                                                                                        SVG_SCRIPT_INVALID,
                                                                                        SVG_RENDER_DIRECTION_INVALID ) )
                            {
                                    fontRenderer->p_grl_shpeng_fp_table = grl_GetShapeng_fptable();
                            }
                            else
                            {
                                /* If Shapeengine init fails, reset autolangdetect state*/
                                fontRenderer->autolangdetect = SVG_FALSE;
                            }
                       }
                    }
                }
                else
                {
                    error = SVG_FREETYPE_ERROR;
        		    SVG_FNT_F("SVG_OUT_OF_MEMORY IN GRLXXCREATEFONTCONTEXT ");
                }
            }
            if (GRL_NO_ERROR == error)
            {
            	EGLenum drawingAPI					= 0;
            	EGLContext	eglContext				= EGL_NO_CONTEXT;
                newContext->p_fontRendererContext   = (void*)fontRenderer;
                newContext->encoding                = SVG_ENCODING_UNICODE;
                newContext->angle                   = 0.0;
                newContext->letter_distance         = 0.0;
                newContext->font_line_space         = 1.0;
                newContext->settings                = SVG_FONT_ANTIALIAS;
                newContext->base_line               = SVG_ORIGIN_BASELINE;
                newContext->error                   = SVG_NO_ERROR;
                newContext->p_grl_fp_table          = &grl_ft;
                newContext->ContourColor            = 0x00;
                newContext->shares_count            = 0;
                newContext->BodyColor               = 0;
                newContext->ContourWidth            = 1;
                newContext->ColorSet                = SVG_FALSE;
                newContext->WidthSet                = SVG_FALSE;
				
				/*check for current bound client API and for a valid context*/
				drawingAPI							= eglQueryAPI();
				eglContext							= eglGetCurrentContext();
				
				if (eglContext == EGL_NO_CONTEXT)
				{
					/*reset drawing API because no context bound*/
					drawingAPI = 0;
				}
				
				switch (drawingAPI)
				{
					case EGL_OPENVG_API:
#ifdef HAS_OPENVG
						fontRenderer->drawingAPI  		= SVG_USE_OPENVG;
#else                	    
                        /* throw error when application wants to use OpenVG, but lib is compiled without */
            		    SVG_FNT_F("OPENVG_NOT_LINKED IN GRLXXCREATEFONTCONTEXT ");
                        error = SVG_EGL_ERROR;
                		fontRenderer->drawingAPI        = SVG_DRAWING_API_MIN;
                        if(( NULL == shareContext) || (NULL == shareContext->p_fontRendererContext))
                        {
                            GRL_free_1D_resource((void*)fontRenderer->shared);
                        }
                        GRL_free_1D_resource((void*)fontRenderer);
		                GRL_free_1D_resource((void*)newContext);
		                newContext = NULL;
#endif						
						break;
					case EGL_OPENGL_ES_API:
#ifdef HAS_OPENGLES
                        fontRenderer->drawingAPI        = SVG_USE_OPENGLES_1;
                        glGetIntegerv(GL_MAX_TEXTURE_SIZE, (GLint*)&fontRenderer->max_texture_size);
#else
                        /* throw error when application wants to use OpenGL, but lib is compiled without */
            		    SVG_FNT_F("OPENGL_NOT_LINKED IN "
            		    		     "GRLXXCREATEFONTCONTEXT ");
                        error = SVG_EGL_ERROR;
                		fontRenderer->drawingAPI        = SVG_DRAWING_API_MIN;
                        if(( NULL == shareContext) || (NULL == shareContext->p_fontRendererContext))
                        {
                            GRL_free_1D_resource((void*)fontRenderer->shared);
                        }
                        GRL_free_1D_resource((void*)fontRenderer);
		                GRL_free_1D_resource((void*)newContext);
		                newContext = NULL;
#endif
                	    break;
                	default:
                		fontRenderer->drawingAPI        = SVG_DRAWING_API_MIN;
            		    SVG_FNT_W("Creating FT font context without "
            		    		 "bound EGL context. No drawing can be done with this context ");
                }
#ifdef HAS_OPENGLES
                /*if gl driver returns 0 or there is no egl context
                 * set the max texture size to 4096*/
                if (fontRenderer->max_texture_size == 0)
                    fontRenderer->max_texture_size = SVG_DEFAUTL_TEXTURE_SIZE;

                SVGUint32 width_height = fontRenderer->max_texture_size;
                SVGUint32 min_cachable_glyphs = 1; /* At least 1 glyph should be cached */
                fontRenderer->max_font_size = ((SVGUint32)(width_height / FILLED_PIXEL_RATIO ) / (SVGUint32)(min_cachable_glyphs + MORE_CHARS_THEN_NEEDED));

				if(SVG_USE_OPENGLES_1 == fontRenderer->drawingAPI)
                {
				    if ( glGetError() != GL_NO_ERROR)
				    {
				        SVG_FNT_W("opengl error in the calling application, check gl errors on the application side");
				    }
				    pszGLExtensions = glGetString(GL_EXTENSIONS);

					/* GL_EXT_multi_draw_arrays */
					if(pszGLExtensions)
					{
						if (strstr((char *)pszGLExtensions, "GL_EXT_multi_draw_arrays"))
						{
							glMultiDrawArraysEXT = (PFNGLMULTIDRAWARRAYSEXTPROC)eglGetProcAddress("glMultiDrawArraysEXT");
						}
					}
				}
#endif
				
#ifdef HAS_OPENGLES2
                if(SVG_USE_OPENGLES_1 == fontRenderer->drawingAPI)
                {

                    fontRenderer->shared->shaderHandle[0] = glCreateShader(GL_VERTEX_SHADER);
                    fontRenderer->shared->shaderHandle[1] = glCreateShader(GL_FRAGMENT_SHADER);
                    fontRenderer->shared->programHandle   = glCreateProgram();
                    if (local_load_binary_shader(&fontRenderer->shared->shaderHandle[0],
                		    	    &fontRenderer->shared->shaderHandle[1]) == SVG_FALSE)
                    {
	                    for(loop = 0; loop < 2 ; loop++)
	                    {
	                        length = (SVGInt32)strlen(shaderFiles[loop]);
	                        /* Supply shader source */
	                        glShaderSource(fontRenderer->shared->shaderHandle[loop], 1, (const char **)&shaderFiles[loop], &length);
	                
	                        /* Compile the shader */
	                        glCompileShader(fontRenderer->shared->shaderHandle[loop]);
	                
	                        /* Check it compiled OK */
	                        glGetShaderiv(fontRenderer->shared->shaderHandle[loop], GL_COMPILE_STATUS, &status);
	                
	                        if (status != GL_TRUE)
	                        {
	                		    SVG_FNT_F("COMPILE_SHADER_FAILED IN "
	                		    		     "GRLXXCREATEFONTCONTEXT ");
	                            error = COMPILE_SHADER_FAILED;          
	                        }
	                    }
	                }
                    if( error == GRL_NO_ERROR)
                    {
                        /* Attach the shader to the programs */
                        glAttachShader(fontRenderer->shared->programHandle, fontRenderer->shared->shaderHandle[0]);
                        glAttachShader(fontRenderer->shared->programHandle, fontRenderer->shared->shaderHandle[1]);
                
                        /* Link the program */
                        glLinkProgram(fontRenderer->shared->programHandle);
                
                        /* Check it linked OK */
                        glGetProgramiv(fontRenderer->shared->programHandle, GL_LINK_STATUS, &status);
                
                        if (status != GL_TRUE)
                        {
                		    SVG_FNT_F("LINK_SHADER_FAILED IN GRLXXCREATEFONTCONTEXT ");
                            error = LINK_SHADER_FAILED;          
                        }
                        else
                        {

#ifdef VALIDATE_NOT_WORKING_FIXED

                            glValidateProgram(fontRenderer->shared->programHandle);
                    
                            glGetProgramiv(fontRenderer->shared->programHandle, GL_VALIDATE_STATUS, &status);
                    
                            if (status != GL_TRUE)
                            {
                    		    SVG_FNT_F("VALIDATE_SHADER_FAILED IN GRLXXCREATEFONTCONTEXT ");
                                error = VALIDATE_SHADER_FAILED;          
                            }
                            else
#endif
                            {
                                fontRenderer->shared->mvplocation     = 
                                    glGetUniformLocation(fontRenderer->shared->programHandle, "mvp");
                                fontRenderer->shared->texturelocation = 
                                    (SVGUint32)glGetAttribLocation(fontRenderer->shared->programHandle, "inputtexcoord");
                                fontRenderer->shared->colorlocation   = 
                                    glGetUniformLocation(fontRenderer->shared->programHandle, "basecolor");
                                fontRenderer->shared->opacitylocation = 
                                    glGetUniformLocation(fontRenderer->shared->programHandle, "opacity");
                                fontRenderer->shared->noaalocation    = 
                                    glGetUniformLocation(fontRenderer->shared->programHandle, "noantialias");
                                fontRenderer->shared->vertexlocation  = 
                                    (SVGUint32)glGetAttribLocation(fontRenderer->shared->programHandle, "position");
    
                                glEnableVertexAttribArray (fontRenderer->shared->vertexlocation);
                                glEnableVertexAttribArray (fontRenderer->shared->texturelocation);
                            }
                        }
                    }
                }
#endif                        

                fontRenderer->additional_space      = 0.0;
                fontRenderer->replacementGlyph      = 0;
                fontRenderer->useReplacementGlyph   = SVG_FALSE;
                fontRenderer->texture               = 0;
            }
            else
            {
                GRL_free_1D_resource((void*)fontRenderer);
                GRL_free_1D_resource((void*)newContext);
                newContext = NULL;
            }
        }
        else
        {
		    SVG_FNT_F("SVG_OUT_OF_MEMORY IN GRLXXCREATEFONTCONTEXT ");
            GRL_free_1D_resource((void*)newContext);
            newContext = NULL;
        }
    }
    else
    {
	    SVG_FNT_F("SVG_OUT_OF_MEMORY IN GRLXXCREATEFONTCONTEXT ");
    }

    return newContext;
}

SVGBoolean grlftDestroyFontContext( SVGFontContext* ctx)
{
    SVGFontFreeTypeRenderer*    p_renderer        = (SVGFontFreeTypeRenderer*)
                                                    ctx->p_fontRendererContext;
    FT_Face                     face              = p_renderer->shared->face;
    SVGBoolean                  destroyed         = SVG_FALSE;

    if(ctx->shares_count == 0)
    {
        __sync_add_and_fetch(&ctx->thread_count, 1);
        local_check_parallel_use( ctx->thread_count );

        destroyed = SVG_TRUE;
        if (ctx->shared == NULL)
        {
            GRL_del_lock(p_renderer->shared->id);

            local_destroy_shared_resources(p_renderer);

            if(p_renderer->stroker_availabe)
            {
                FT_Stroker_Done( p_renderer->shared->stroker );
            }
            /*delete hash table */
            UTIL_del_hsh(p_renderer->shared->hash_tbl_strk);
            UTIL_del_hsh(p_renderer->shared->hash_tbl_fill);

            local_clear_cache(p_renderer, SVG_FALSE);/*clear stroked cache*/
			local_clear_cache(p_renderer, SVG_TRUE);/*clear filled cache*/

            if(SVG_TRUE == ctx->validFont)
            {
                if(0 != FT_Done_Face(face))
                {
            	    SVG_FNT_E("FACE_DONE_FAIL IN GRLXXDESTROYFONTCONTEXT");
                }
            }
            if(0 != FT_Done_FreeType( p_renderer->shared->library ))
            {
        	    SVG_FNT_E("FREETYPE_DONE_FAIL IN GRLXXDESTROYFONTCONTEXT");
            }
            /* throw away the wallpapers */
			local_free_wallpapers( ctx );

#ifdef HAS_OPENGLES2
			if (SVG_USE_OPENGLES_1 == p_renderer->drawingAPI)
			{
				glDeleteShader(p_renderer->shared->shaderHandle[0]);
				glDeleteShader(p_renderer->shared->shaderHandle[1]);
				glDeleteProgram(p_renderer->shared->programHandle);
			}
#endif      
			if(p_renderer->autolangdetect == SVG_TRUE)
			{
                /* Free the shapeeng object and reset all properties*/
                grl_ShapeEng_Plugout(p_renderer->shared->p_shape_eng);
                p_renderer->shared->p_shape_eng = NULL;
                p_renderer->p_grl_shpeng_fp_table = NULL;
			}
            GRL_free_1D_resource((void*)p_renderer->shared);
        }
        else
        {
            GRL_wai_lock(p_renderer->shared->id);
            /* decrement shares count in shared context */
            ctx->shared->shares_count--;
            GRL_sig_lock(p_renderer->shared->id);
        }

        p_renderer->autolangdetect = SVG_FALSE;

        local_check_parallel_use( ctx->thread_count );
        __sync_sub_and_fetch(&ctx->thread_count, 1);
        
        /* and now throw away not shared resources */
        GRL_free_1D_resource((void*)ctx->p_fontRendererContext);
        GRL_free_1D_resource((void*)ctx);
    }
    else
    {
	    SVG_FNT_E("CONTEXT_IN_USE IN GRLXXDESTROYFONTCONTEXT ");
    }
    return (destroyed);
}

void grlftSetFontSize( SVGFontContext* ctx,
                       SVGUint32 fontSize)
{
    SVGUint32                   loop              = 0;
    SVGUint32                   delta             = 0xffffffff;
    SVGUint32                   font_size         = 0;
    SVGFontFreeTypeRenderer*    p_renderer        = (SVGFontFreeTypeRenderer*)
                                                    ctx->p_fontRendererContext;
    SVGFont*                    font              = p_renderer->shared->p_svgFont;
    
    /* p_renderer NULL cross-check, already checked in grlftCreateFontContext */
    if(p_renderer == NULL)
    {
	    SVG_FNT_F("SVG_POINTER_NULL IN GRLXXSETFONTSIZE ");
        grlftSetError(ctx, SVG_POINTER_NULL);
        return;
    }

    if(p_renderer->max_font_size < fontSize)
    {
	    SVG_FNT_E("SVG_INVALID_VALUE IN GRLXXSETFONTSIZE ");
        grlftSetError(ctx, SVG_INVALID_VALUE);
    }
    else
    {
        if(font->lengthSizes > 0)
        {
            for(loop = 0; loop < font->lengthSizes; loop++)
            {
                if((SVGUint32)abs(font->allowedSizes[loop] - (SVGUint8)fontSize) < delta)
                {
                    delta = (SVGUint32)abs(font->allowedSizes[loop] - (SVGUint8)fontSize);
                    font_size = font->allowedSizes[loop];
                    grlftSetError(ctx, SVG_INVALID_VALUE);
                }
            }
        }
        else
        {
            font_size = fontSize;
        }

        if (font_size > SVG_MAX_FONT_SIZE)
        {
    	    SVGUint32 glyphs_per_row    = (SVGUint32)sqrt(p_renderer->cachable_glyphs);
            SVGUint32 width_height = (glyphs_per_row + MORE_CHARS_THEN_NEEDED ) * (SVGUint32)((SVGFloat)(SVG_MAX_FONT_SIZE) * FILLED_PIXEL_RATIO);
	    p_renderer->cachable_glyphs = ((SVGUint32)((width_height / FILLED_PIXEL_RATIO) / ((SVGFloat)(font_size))) - MORE_CHARS_THEN_NEEDED);
            p_renderer->cachable_glyphs *= p_renderer->cachable_glyphs;
			SVG_FNT_E("%d IN GRL_REDUCE_CACHABLE_GLYPHS",p_renderer->cachable_glyphs);

    	}

        if(NULL != ctx->shared)
        {
            GRL_wai_lock(p_renderer->shared->id);
        }
        __sync_add_and_fetch(&ctx->thread_count, 1);//atomic incrementation, used for parallel access detection
        local_check_parallel_use( ctx->thread_count );
        /* throw away cache structures when new font size is bigger then before */
        if(p_renderer->font_size < font_size)
        {
        	 /*clear hash table */
        	 UTIL_clr_hsh(p_renderer->shared->hash_tbl_strk);
        	 UTIL_clr_hsh(p_renderer->shared->hash_tbl_fill);

            local_clear_cache(p_renderer, SVG_FALSE);/*clear stroked cache*/
            local_clear_cache(p_renderer, SVG_TRUE);/*clear filled cache*/

            /* throw away the wallpapers */
            local_free_wallpapers( ctx );

            p_renderer->shared->cache_strk = NULL;
            p_renderer->shared->cache_fill = NULL;
            p_renderer->shared->cache_count_strk = 0;
            p_renderer->shared->cache_count_fill = 0;

            p_renderer->shared->cache_count = &p_renderer->shared->cache_count_fill;
            p_renderer->shared->cache = &p_renderer->shared->cache_fill;

            local_create_wallpapers( ctx , fontSize );

        }
        p_renderer->font_size = font_size;
        if(0 != FT_Set_Pixel_Sizes(p_renderer->shared->face, 0, p_renderer->font_size ))
        {
			SVG_FNT_E("SET_PIXEL_SIZE_FAIL IN GRLXXSETFONTSIZE");

            grlftSetError(ctx, SET_PIXEL_SIZE_FAIL);
        }

        p_renderer->ascender            = 0.0;
        p_renderer->descender           = 0.0;
        p_renderer->underline_position  = 0.0;
        p_renderer->underline_thickness = 0.0;
        ctx->font_line_space = (SVGUint32)
            p_renderer->shared->face->size->metrics.y_ppem;

        if (0 != p_renderer->shared->face->units_per_EM)
        {
            p_renderer->left              =
                (SVGUint16)abs((p_renderer->shared->face->bbox.xMin *
                p_renderer->shared->face->size->metrics.x_ppem) /
                p_renderer->shared->face->units_per_EM);
            p_renderer->right              =
                (SVGUint16)abs((p_renderer->shared->face->bbox.xMax *
                p_renderer->shared->face->size->metrics.x_ppem) /
                p_renderer->shared->face->units_per_EM);
            p_renderer->ascender              =
                (SVGUint16)abs((p_renderer->shared->face->bbox.yMax *
                p_renderer->shared->face->size->metrics.y_ppem) /
                p_renderer->shared->face->units_per_EM);
            p_renderer->descender             =
                (SVGUint16)abs((p_renderer->shared->face->bbox.yMin *
                p_renderer->shared->face->size->metrics.y_ppem) /
                p_renderer->shared->face->units_per_EM);
            p_renderer->underline_position    =
                (SVGUint16)abs((p_renderer->shared->face->underline_position *
                p_renderer->shared->face->size->metrics.y_ppem) /
                p_renderer->shared->face->units_per_EM);
            p_renderer->underline_thickness   =
                (SVGUint16)((p_renderer->shared->face->underline_thickness *
                p_renderer->shared->face->size->metrics.y_ppem) /
                p_renderer->shared->face->units_per_EM);
            ctx->font_line_space = (SVGUint32)
                ((SVGFloat)((abs(p_renderer->shared->face->bbox.yMax) +
                  abs(p_renderer->shared->face->bbox.yMin)) *
                  p_renderer->shared->face->size->metrics.y_ppem) /
                  (SVGFloat)p_renderer->shared->face->units_per_EM);
        }

        local_check_parallel_use( ctx->thread_count );
        __sync_sub_and_fetch(&ctx->thread_count, 1);//atomic derementation, used for parallel access detection
        if(NULL != ctx->shared)
        {
            GRL_sig_lock(p_renderer->shared->id);
        }
        /*added to clear the state of egl, textures and memory*/
        eglWaitClient();
    }
}

void grlftLoadFont( SVGFontContext* ctx,
                    SVGFont* font,
                    SVGEncoding charMap,
                    SVGUint32 fontSize)
{
    SVGUint8*                   address           = NULL;
    SVGUint32                   len               = 0;
    FT_Error                    error             = 0;
    SVGUint32                   loop              = 0;
    SVGUint32                   delta             = 0xffffffff;
    SVGUint32                   font_size         = 0;
    SVGFontFreeTypeRenderer*    p_renderer        = (SVGFontFreeTypeRenderer*)
                                                     ctx->p_fontRendererContext;
    SVGBoolean                  validFont         = SVG_FALSE;

    /* p_renderer NULL cross-check, already checked in grlftCreateFontContext */
    if(p_renderer == NULL)
    {
	    SVG_FNT_F("SVG_POINTER_NULL IN GRLXXLOADFONT ");
        grlftSetError(ctx, SVG_POINTER_NULL);
        return;
    }

    if(fontSize > p_renderer->max_font_size)
    {
        grlftSetError(ctx, SVG_INVALID_VALUE);
	    SVG_FNT_E("SVG_INVALID_VALUE IN GRLXXLOADFONT ");
    }
    else
    {

        if (NULL == ctx->shared)
        {
            validFont = ctx->validFont;
        }
        else
        {
            validFont = ctx->shared->validFont;
            GRL_wai_lock(p_renderer->shared->id);
        }
        __sync_add_and_fetch(&ctx->thread_count, 1);//atomic incrementation, used for parallel access detection
        local_check_parallel_use( ctx->thread_count );

        /* get renderer information */
        p_renderer->useReplacementGlyph = SVG_FALSE;
        p_renderer->replacementGlyph    = 0;

        if ((charMap < SVG_ENCODING_ASCII) || (charMap >= SVG_ENCODING_MAX))
        {
            grlftSetError(ctx, SVG_INVALID_ENUM);
        }
        else
        {
            if(font->lengthSizes > 0)
            {
                for(loop = 0; loop < font->lengthSizes; loop++)
                {
                    if((SVGUint32)abs(font->allowedSizes[loop] - (SVGUint8)fontSize) < delta)
                    {
                        delta = (SVGUint32)abs(font->allowedSizes[loop] - (SVGUint8)fontSize);
                        font_size = font->allowedSizes[loop];
                    }
                }
                if(font_size != fontSize)
                {
                    grlftSetError(ctx, SVG_INVALID_VALUE);
                }
            }
            else
            {
                font_size = fontSize;
            }
            ctx->encoding = charMap;
            ctx->font_line_space = 0;


            /* throw away previous face and font resources when it was valid and when
               the font has changed */
            if((SVG_TRUE == validFont) &&
               ((p_renderer->shared->p_svgFont != GRL_get_font_id(ctx, font)) ||
                (p_renderer->font_size < font_size)))
            {
                 /*delete hash table */
                UTIL_clr_hsh(p_renderer->shared->hash_tbl_strk);
                UTIL_clr_hsh(p_renderer->shared->hash_tbl_fill);

                local_clear_cache(p_renderer, SVG_FALSE);/*clear stroked cache*/
                local_clear_cache(p_renderer, SVG_TRUE);/*clear filled cache*/

                /* throw away the wallpapers */
                local_free_wallpapers( ctx );

                p_renderer->shared->cache_fill = NULL;
                p_renderer->shared->cache_strk = NULL;
                p_renderer->shared->cache_count_fill = 0;
                p_renderer->shared->cache_count_strk = 0;

                p_renderer->shared->cache_count = &p_renderer->shared->cache_count_fill;
                p_renderer->shared->cache = &p_renderer->shared->cache_fill;

                if(0 != FT_Done_Face(p_renderer->shared->face))
                {
					SVG_FNT_E("FACE_DONE_FAIL IN GRLXXLOADFONT");

                }

                if(p_renderer->autolangdetect== SVG_TRUE)
                {
                    p_renderer->p_grl_shpeng_fp_table->DeleteFont((void*)p_renderer->shared->p_shape_eng);
                }

                /* set context to invalid to ensure that access is only available
                   after correct font loading */
                if (NULL == ctx->shared)
                {
                    ctx->validFont = SVG_FALSE;
                }
                else
                {
                    ctx->shared->validFont = SVG_FALSE;
                }
                validFont = SVG_FALSE;
            }

            /* load the new font: either a font has never be loaded before or  it
               was "erased" due to another font should be loaded */
            if(SVG_FALSE == validFont)
            {

                if(NULL == strstr((SVGChar*)font->filename, (SVGChar*)"/flashfont/"))
                {
                    if(0 != FT_New_Face(p_renderer->shared->library,
                                        (SVGChar*)font->filename,
                                        (FT_Long) font->faceIndex, &p_renderer->shared->face))
                    {
						SVG_FNT_E("OPEN_FILE_FONT_ERROR IN GRLXXLOADFONT");
                        error = OPEN_FILE_FONT_ERROR;
                        grlftSetError(ctx, error);
                    }
                }
                else
                {
                    /* first split the "filename" into address and len ...*/
                    memcpy(&address, (&font->filename[11]), 4);
                    memcpy(&len, (&font->filename[16]), 4);
                    /* ... and then open the font */
                    if(0 != FT_New_Memory_Face(p_renderer->shared->library,
                                               address,
                                               (SVGInt64)len, (FT_Long)font->faceIndex, &p_renderer->shared->face))
                    {
						SVG_FNT_E("OPEN_FILE_FONT_ERROR IN GRLXXLOADFONT");
                        error = OPEN_FILE_FONT_ERROR;
                        grlftSetError(ctx, error);
                    }
                }
            }

            if(error == 0)
            {
                if(0 != FT_Set_Pixel_Sizes(p_renderer->shared->face, 0, font_size ))
                {
                    error = SET_PIXEL_SIZE_FAIL;
					SVG_FNT_E("SET_PIXEL_SIZE_FAIL IN GRLXXLOADFONT");

                    grlftSetError(ctx, error);
                }
                p_renderer->font_size = font_size;


            }

            if (error == 0)
            {
                if (local_set_font( ctx, font, charMap, p_renderer) == SVG_FALSE)
                {
                    grlftSetError(ctx, SVG_FREETYPE_ERROR);
					SVG_FNT_F("SVG_INVALID_OPERATION IN GRLXXLOADFONT");
                }
                else
                {
                	if (font_size > SVG_MAX_FONT_SIZE)
                	{
                		SVGUint32 glyphs_per_row    = (SVGUint32)sqrt(p_renderer->cachable_glyphs);
                		SVGUint32 width_height = (glyphs_per_row + MORE_CHARS_THEN_NEEDED ) * (SVGUint32)((SVGFloat)(SVG_MAX_FONT_SIZE) * FILLED_PIXEL_RATIO);
				p_renderer->cachable_glyphs = ((SVGUint32)((width_height / FILLED_PIXEL_RATIO) / ((SVGFloat)(font_size))) - MORE_CHARS_THEN_NEEDED);
				p_renderer->cachable_glyphs *= p_renderer->cachable_glyphs;
						SVG_FNT_W("%d GRL_REDUCE_CACHABLE_GLYPHS",p_renderer->cachable_glyphs);

                	}

                	ctx->font_line_space = (SVGUint32)
                        p_renderer->shared->face->size->metrics.y_ppem;
                    p_renderer->ascender            = 0.0;
                    p_renderer->descender           = 0.0;
                    p_renderer->underline_position  = 0.0;
                    p_renderer->underline_thickness = 0.0;
                    if(0 != p_renderer->shared->face->units_per_EM)
                    {
                        ctx->font_line_space = (SVGUint32)
                            ((SVGFloat)((abs(p_renderer->shared->face->bbox.yMax) +
                              abs(p_renderer->shared->face->bbox.yMin)) *
                              p_renderer->shared->face->size->metrics.y_ppem) /
                              (SVGFloat)p_renderer->shared->face->units_per_EM);
                        p_renderer->left              =
                            (SVGUint16)abs((p_renderer->shared->face->bbox.xMin *
                            p_renderer->shared->face->size->metrics.x_ppem) /
                            p_renderer->shared->face->units_per_EM);
                        p_renderer->right              =
                            (SVGUint16)abs((p_renderer->shared->face->bbox.xMax *
                            p_renderer->shared->face->size->metrics.x_ppem) /
                            p_renderer->shared->face->units_per_EM);
                        p_renderer->ascender              =
                            (SVGUint16)abs((p_renderer->shared->face->bbox.yMax *
                            p_renderer->shared->face->size->metrics.y_ppem) /
                            p_renderer->shared->face->units_per_EM);
                        p_renderer->descender             =
                            (SVGUint16)abs((p_renderer->shared->face->bbox.yMin *
                            p_renderer->shared->face->size->metrics.y_ppem) /
                            p_renderer->shared->face->units_per_EM);
                        p_renderer->underline_position    =
                            (SVGUint16)abs((p_renderer->shared->face->underline_position *
                            p_renderer->shared->face->size->metrics.y_ppem) /
                            p_renderer->shared->face->units_per_EM);
                        p_renderer->underline_thickness   =
                            (SVGUint16)((p_renderer->shared->face->underline_thickness *
                            p_renderer->shared->face->size->metrics.y_ppem) /
                            p_renderer->shared->face->units_per_EM);
                    }
                    /* create wallpaper texture */
    				local_create_wallpapers( ctx , fontSize );
                }
            }

               if(p_renderer->autolangdetect == SVG_TRUE)
              {
                    if(error == 0)
                    {
                        if(0 != p_renderer->p_grl_shpeng_fp_table->CreateFont(
                                                         (void*)p_renderer->shared->p_shape_eng,
                                                          p_renderer->shared->face))
                        {
							SVG_FNT_E("HB_FONT_HNDL_CREATION_FAILED IN "
									"GRLXXLOADFONT");
                            error = HB_FONT_HNDL_CREATION_FAILED;
                            grlftSetError(ctx, error);
                        }
                    }
            }

            /*added to clear the state of egl, textures and memory*/
            eglWaitClient();
        }
        local_check_parallel_use( ctx->thread_count );
        __sync_sub_and_fetch(&ctx->thread_count, 1);//atomic derementation, used for parallel access detection
        if(NULL != ctx->shared)
        {
            GRL_sig_lock(p_renderer->shared->id);
        }
    }
}

void grlftEnableFontSettings( SVGFontContext* ctx, SVGUint32 settings)
{
    SVGFontFreeTypeRenderer*    p_renderer   = (SVGFontFreeTypeRenderer*)
                                                ctx->p_fontRendererContext;

    if((settings < SVG_FONT_HINTING) || (settings > SVG_FONT_INVALID_SETTING))
    {
	    SVG_FNT_W("SVG_INVALID_VALUE IN GRLXXENABLEFONTSETTINGS");
        grlftSetError (ctx, SVG_INVALID_VALUE);
    }
    else
    {
        if(NULL != ctx->shared)
        {
            GRL_wai_lock(p_renderer->shared->id);
        }
        __sync_add_and_fetch(&ctx->thread_count, 1);//atomic incrementation, used for parallel access detection
        local_check_parallel_use( ctx->thread_count );

        if ((settings & SVG_FONT_KERNING) != 0)
        {
            if (FT_HAS_KERNING((SVGUint32)((SVGFontFreeTypeRenderer *)
                (ctx->p_fontRendererContext))->shared->face) != 0)
            {
                ctx->settings |= SVG_FONT_KERNING;
            }
            else
            {
                grlftSetError (ctx, SVG_INVALID_OPERATION);
            }
        }

        /* vertical should be available for all fonts */
        if ((settings & SVG_FONT_VERTICAL) != 0)
        {
            ctx->settings |= SVG_FONT_VERTICAL;
        }

        if ((settings & SVG_FONT_HINTING) != 0)
        {
            ctx->settings |= SVG_FONT_HINTING;
        }

        if ((settings & SVG_FONT_ANTIALIAS) != 0)
        {
            ctx->settings |= SVG_FONT_ANTIALIAS;
        }

        if ((settings & SVG_FONT_2COLOR) != 0)
        {
            ctx->settings |= SVG_FONT_2COLOR;
			/*create necessary wallpapers (filled or stroked)*/
			local_create_wallpapers( ctx , p_renderer->font_size );
		}

        if ((settings & SVG_FONT_OUTLINE) != 0)
        {
            ctx->settings |= SVG_FONT_OUTLINE;
			/*create necessary wallpapers (stroked)*/
			local_create_wallpapers( ctx , p_renderer->font_size );
        }

        if ((settings & SVG_FONT_NO_ENCLOSING_SPACES) != 0)
        {
            ctx->settings |= SVG_FONT_NO_ENCLOSING_SPACES;
        }

        if ((settings & SVG_FONT_USE_REPLACEMENT_CHAR) != 0)
        {
            p_renderer->useReplacementGlyph = SVG_TRUE;
        }
        if ((settings & SVG_FONT_OUTLINE_SPACE_EXTENSION) != 0)
        {
            ctx->settings |= SVG_FONT_OUTLINE_SPACE_EXTENSION;
        }
        if ((settings & SVG_FONT_OPACITY) != 0)
        {
            ctx->settings |= SVG_FONT_OPACITY;
        }
        if ((settings & SVG_FONT_USE_CURRENT_MATRIX) != 0)
        {
            ctx->settings |= SVG_FONT_USE_CURRENT_MATRIX;
        }
        if ((settings & SVG_FONT_SCALE) != 0)
        {
            ctx->settings |= SVG_FONT_SCALE;
        }
        if ((settings & SVG_FONT_USE_CURRENT_PROGRAM) != 0)
        {
            ctx->settings |= SVG_FONT_USE_CURRENT_PROGRAM;
        }
        if ((settings & SVG_FONT_MIPMAPPING) != 0)
        {
            if (SVG_USE_OPENVG == p_renderer->drawingAPI)
            {
                /* trace information that mipmapping is not supported for OpenVG */
        	    SVG_FNT_W("GRL_MIPMAPPING_NOT_SUPPORTED IN GRLXXENABLEFONTSETTINGS");
            }
            else
            {
                ctx->settings |= SVG_FONT_MIPMAPPING;
#ifdef HAS_OPENGLES 
				if( p_renderer->shared->glWallpaperStrk != 0)
				{
					glBindTexture(GL_TEXTURE_2D, p_renderer->shared->glWallpaperStrk);
	#ifdef HAS_OPENGLES1
					glTexEnvx(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
					glTexParameteri(GL_TEXTURE_2D, GL_GENERATE_MIPMAP, GL_TRUE);
	#else
					glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
	#endif
				}
				if( p_renderer->shared->glWallpaperFill != 0 )
				{
					glBindTexture(GL_TEXTURE_2D, p_renderer->shared->glWallpaperFill);
	#ifdef HAS_OPENGLES1
					glTexEnvx(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
					glTexParameteri(GL_TEXTURE_2D, GL_GENERATE_MIPMAP, GL_TRUE);
	#else
					glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
	#endif
				}
#endif
            }
        }

        local_check_parallel_use( ctx->thread_count );
        __sync_sub_and_fetch(&ctx->thread_count, 1);//atomic incrementation, used for parallel access detection
        if(NULL != ctx->shared)
        {
            GRL_sig_lock(p_renderer->shared->id);
        }
    }
}

void grlftDisableFontSettings( SVGFontContext* ctx, SVGUint32 settings)
{
    SVGFontFreeTypeRenderer*    p_renderer   = (SVGFontFreeTypeRenderer*)
                                                ctx->p_fontRendererContext;
    if((settings < SVG_FONT_HINTING) || (settings > 0xcffff))
    {
	    SVG_FNT_W("SVG_INVALID_VALUE IN GRLXXDISABLEFONTSETTINGS");
        grlftSetError (ctx, SVG_INVALID_VALUE);
    }
    else
    {
        if(NULL != ctx->shared)
        {
            GRL_wai_lock(p_renderer->shared->id);
        }
        __sync_add_and_fetch(&ctx->thread_count, 1);//atomic incrementation, used for parallel access detection
        local_check_parallel_use( ctx->thread_count );

        ctx->settings &= ~settings;
        
		if ((settings & SVG_FONT_USE_REPLACEMENT_CHAR) != 0)
        {
            p_renderer->useReplacementGlyph = SVG_FALSE;
        }
		if ((settings & SVG_FONT_OUTLINE) != 0)
        {
			/*create necessary wallpapers (filled) if only outline was set*/
			local_create_wallpapers(ctx , p_renderer->font_size );
		}
		
		if( (settings & SVG_FONT_MIPMAPPING) != 0)
		{
#ifdef HAS_OPENGLES 
			if( (p_renderer->shared->glWallpaperStrk != 0) )
			{
				glBindTexture(GL_TEXTURE_2D, p_renderer->shared->glWallpaperStrk);
	#ifdef HAS_OPENGLES1
				glTexEnvx(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
				glTexParameteri(GL_TEXTURE_2D, GL_GENERATE_MIPMAP, GL_FALSE);
	#else
				glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR );
				glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR );
	#endif
			}
			if( p_renderer->shared->glWallpaperFill != 0 )
			{
				glBindTexture(GL_TEXTURE_2D, p_renderer->shared->glWallpaperFill);
	#ifdef HAS_OPENGLES1
				glTexEnvx(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
				glTexParameteri(GL_TEXTURE_2D, GL_GENERATE_MIPMAP, GL_FALSE);
	#else
				glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR );
				glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR );					
	#endif
			}
#endif
		}

		local_check_parallel_use( ctx->thread_count );
		__sync_sub_and_fetch(&ctx->thread_count, 1);
        if(NULL != ctx->shared)
        {
         GRL_sig_lock(p_renderer->shared->id);
        }
    }
}

SVGBoolean grlftIsFontSettingEnabled( SVGFontContext* ctx,
                                      SVGUint32 setting)
{
    SVGBoolean      ret = SVG_TRUE;

    if((setting < SVG_FONT_HINTING) || (setting > 0xcffff))
    {
	    SVG_FNT_W("SVG_INVALID_VALUE IN GRLXXISFONTSETTINGENABLED");
        grlftSetError (ctx, SVG_INVALID_VALUE);
        ret = SVG_FALSE;
    }
    else
    {
        if ((ctx->settings & setting) == 0)
        {
            ret = SVG_FALSE;
        }
    }

    return ret;
}


SVGUint32 grlftGetFontBitmapExt( SVGFontContext* ctx,
                                 SVGUint8**      text,
                                 SVGUint32*      textLength,
                                 SVGUint32	   	 textNumber,
                                 SVGUint32*      filledTexture,
								 SVGUint32*      strokedTexture,											
                                 SVGBBox**       filledTextureBox,
                                 SVGBBox**       strokedTextureBox,
                                 SVGBBox**       filledGlyphBox,
                                 SVGBBox**       strokedGlyphBox,
                                 SVGUint32*      glyphNumber)
{
    SVGFontFreeTypeRenderer* p_renderer     = (SVGFontFreeTypeRenderer*)
                                               ctx->p_fontRendererContext;
    SVGUint32                glyphs_to_draw = 0;
    SVGUint32                loop           = 0;
    SVGUint32                hbGlyphCount	   = 0;
    get_char_info_struct     char_info      = {NULL, 0, 0, 0, 0, 0, {0,0}, SVG_ENCODING_MIN, 0, 0, 0, 0, NULL, 0};
    SVGError                 error          = SVG_NO_ERROR;
    SVGBoolean               term_found     = SVG_FALSE;
    SVGUint32                l_index          = 1; /* fill */
    SVGUint32                stroke_width   = 0;
    SVGBBox                  **glyphbox      = filledGlyphBox;
    SVGBBox                  **texturebox    = filledTextureBox;
    SVGFloat                 distance       = ctx->letter_distance;    
    SVGUint32                textloop       = 0;
    SVGUint8*                theText        = NULL;
	SVGUint32				 start			= 1;
	SVGUint32				 end			= 2;
	SVGFloat				 *max_gap		= (SVGFloat*)GRL_malloc_1D_resource(sizeof(SVGFloat)*textNumber);
	SVGFloat				 absl_offset_y  = 0; 
	SVGLangScript               script_tag = SVG_TAG_NONE;
	SVGRenderDirection      rendr_direc_Tag = SVG_RENDER_DIRECTION_INVALID;
	SVGUint32                   *p_script_rendr_data = NULL;
	SVGUint32                   *p_temp_scriptrendrdata = NULL;
#ifdef HAS_OPENGLES1
    SVGFloat                 stroke_width_f = 0.0f;
#endif    
	SVGBoolean				add_new_glyph	= SVG_FALSE;
	SVGBoolean              is_update_fill     = SVG_FALSE; /*indicates texture update of fill glyphs*/
	SVGBoolean              is_update_strk     = SVG_FALSE; /*indicates texture update of strk glyphs*/

    p_script_rendr_data = (SVGUint32*)GRL_malloc_1D_resource((sizeof(script_tag)*2)*textNumber);

    if( p_script_rendr_data != NULL)
    {
        memset(p_script_rendr_data,0,((sizeof(script_tag)*2)*textNumber));
    }

    /* lock shared context now due to changes in freetype settings */
	if(NULL != ctx->shared)
    {
        GRL_wai_lock(p_renderer->shared->id);
    }
    __sync_add_and_fetch(&ctx->thread_count, 1);//atomic incrementation, used for parallel access detection
    local_check_parallel_use( ctx->thread_count );

    p_renderer->distance_x = 0.0f;
    p_renderer->distance_y = 0.0f;
    char_info.encoding     = ctx->encoding;
	
	if( max_gap != NULL)
	{
		memset(max_gap,0,sizeof(SVGFloat)*textNumber);

	#ifdef HAS_OPENVG
		if(SVG_USE_OPENVG == p_renderer->drawingAPI)
		{
			p_renderer->wallpaperWidth  = p_renderer->shared->vgWallpaperWidth;
			p_renderer->wallpaperHeight = p_renderer->shared->vgWallpaperHeight;
			vgSeti(VG_MATRIX_MODE, VG_MATRIX_IMAGE_USER_TO_SURFACE);
		}
	#endif
	#ifdef HAS_OPENGLES
		if(SVG_USE_OPENGLES_1 == p_renderer->drawingAPI)
		{     
			p_renderer->wallpaperWidth  = p_renderer->shared->glWallpaperWidth;
			p_renderer->wallpaperHeight = p_renderer->shared->glWallpaperHeight;
			glGetIntegerv( GL_UNPACK_ALIGNMENT, &(p_renderer->glUnpackAlignment) );
		}
	#endif				
		if (((ctx->settings & (SVG_FONT_2COLOR | SVG_FONT_OUTLINE)) != 0) && 
			(SVG_TRUE == p_renderer->stroker_availabe))
		{
		/*setup the loop to run two times,for filled and stroked glyphs(2color)*/
			start = 0;
			end = 2;
			if(SVG_TRUE == ctx->WidthSet )
			{
				stroke_width = ctx->ContourWidth;
			}
			else
			{
	#ifdef HAS_OPENVG
				if(SVG_USE_OPENVG == p_renderer->drawingAPI)
				{
					stroke_width = (SVGUint32)vgGeti( VG_STROKE_LINE_WIDTH );
				}
	#endif                        
	#ifdef HAS_OPENGLES
				if(SVG_USE_OPENGLES_1 == p_renderer->drawingAPI)
				{ 
	#ifdef HAS_OPENGLES1
					glGetFloatv(GL_LINE_WIDTH, &stroke_width_f);
					stroke_width = (SVGUint32)stroke_width_f;
	#else
					stroke_width = ctx->ContourWidth;
	#endif              
				}
	#endif            
			}
			if ((ctx->settings & SVG_FONT_OUTLINE_SPACE_EXTENSION) != 0)
			{
				ctx->letter_distance += (SVGFloat) stroke_width;
			}
			if(stroke_width < 2)
			{
				stroke_width = 2;
			}
			FT_Stroker_Set(p_renderer->shared->stroker,
						   (SVGInt64)stroke_width << 5,
						   FT_STROKER_LINECAP_ROUND,
						   FT_STROKER_LINEJOIN_MITER,
						   (SVGInt64)3 << 15);
		}

		if( ((ctx->settings & SVG_FONT_OUTLINE) != 0) &&
			((ctx->settings & SVG_FONT_2COLOR) == 0))
		{/*setup the loop to run only ones, for stroked glyphs (just outline )*/
			start = 0;
			end = 1;
		}
		
		if( (p_renderer->stroke_width != stroke_width) &&
			(p_renderer->stroke_width != 0) &&
			(stroke_width != 0))
		{
			local_del_strk_glyphs( ctx );
		}
        p_renderer->stroke_width = stroke_width;
		
		p_renderer->distance_x = ctx->letter_distance;
        p_renderer->is_vertical = SVG_FALSE;

		if ((ctx->settings & SVG_FONT_VERTICAL) == SVG_FONT_VERTICAL)
		{
			p_renderer->is_vertical = SVG_TRUE;
			p_renderer->distance_y = ctx->letter_distance;
			p_renderer->distance_x = 0.0f;
		}
		
		for(l_index = start; l_index < end; l_index++)
		{
                    p_temp_scriptrendrdata = p_script_rendr_data;
		    add_new_glyph = SVG_FALSE;
			if( l_index == 0 )
			{
#ifdef HAS_OPENGLES
				if(SVG_USE_OPENGLES_1 == p_renderer->drawingAPI)
				{
					glBindTexture(GL_TEXTURE_2D,p_renderer->shared->glWallpaperStrk);
					p_renderer->row = p_renderer->shared->glRowStrk;
				}
#endif
#ifdef HAS_OPENVG
				if(SVG_USE_OPENVG == p_renderer->drawingAPI)
				{
					p_renderer->row = p_renderer->shared->vgRowStrk;
				}
#endif
				p_renderer->shared->cache = &p_renderer->shared->cache_strk;
				p_renderer->shared->cache_count = &p_renderer->shared->cache_count_strk;
				p_renderer->shared->hash_tbl_glyphs = p_renderer->shared->hash_tbl_strk;
				
				glyphbox      = strokedGlyphBox;
				texturebox    = strokedTextureBox;
			}
			else
			{
#ifdef HAS_OPENGLES
				if(SVG_USE_OPENGLES_1 == p_renderer->drawingAPI)
				{
					glBindTexture(GL_TEXTURE_2D,p_renderer->shared->glWallpaperFill);
					p_renderer->row = p_renderer->shared->glRowFill;
				}
#endif				
#ifdef HAS_OPENVG
				if(SVG_USE_OPENVG == p_renderer->drawingAPI)
				{
					p_renderer->row = p_renderer->shared->vgRowFill;
				}
#endif
				p_renderer->shared->cache = &p_renderer->shared->cache_fill;
				p_renderer->shared->cache_count = &p_renderer->shared->cache_count_fill;
				p_renderer->shared->hash_tbl_glyphs = p_renderer->shared->hash_tbl_fill;
				
				glyphbox      = filledGlyphBox;
				texturebox    = filledTextureBox;
			}
			
			for(textloop = 0; textloop < textNumber; textloop++)
			{
                    /*
                      * Finding scripts for the passed string
                      * if the complex scripts is found then enable the shaping engine
                      * If not found,the simple string is processed without shaping engine
                      *
                      * Finding scripts is only one time executed for stroke enabled strings and
                      * formed the scripts list in the buffers which will be refered for filled enabled
                      * strings. It will avoid duplicate searching for same strings
                      */
                           if((SVG_TRUE == p_renderer->autolangdetect) && \
                                    ( SVG_TRUE == local_shpeng_get_autolang_config()))
                           {
                                if( NULL != p_temp_scriptrendrdata)
                                {
                                      if( 0 == p_temp_scriptrendrdata[0] )
                                      {
                                          p_temp_scriptrendrdata[0] = 1;
                                          p_temp_scriptrendrdata[1] = 1;
                                          char_info.theChar     = text[textloop];
                                          char_info.textlength = textLength[textloop];
                                           if( SVG_TRUE == local_shpeng_find_scripttag( &char_info,
                                                                                                                   (SVGUint32*)&script_tag,
                                                                                                                   (SVGUint32*)&rendr_direc_Tag) )
                                           {
                                                if(SVG_NO_ERROR ==  local_shpeng_set_langscript_rendrdirection(p_renderer,
                                                                                                                      (SVGLangScript)script_tag,
                                                                                                                      (SVGRenderDirection)rendr_direc_Tag) )
                                              {
                                                    p_temp_scriptrendrdata[0] = script_tag;
                                                    p_temp_scriptrendrdata[1] = rendr_direc_Tag;
                                                  p_renderer->enableshapingeng = SVG_TRUE;
                                                  p_renderer->script_tag = script_tag;
                                              }
                                           }
                                       }
                                       else if(p_temp_scriptrendrdata[0] != 1)
                                       {
                                             if(SVG_NO_ERROR ==  local_shpeng_set_langscript_rendrdirection(p_renderer,
                                                                                                                   (SVGLangScript)p_temp_scriptrendrdata[0] ,
                                                                                                                   (SVGRenderDirection)p_temp_scriptrendrdata[1] ))
                                           {
                                               p_renderer->enableshapingeng = SVG_TRUE;
                                               p_renderer->script_tag = script_tag;
                                           }
                                       }
                                       p_temp_scriptrendrdata += 2;
                                  }
                             }

				char_info.prevGlyph   = 0;
				char_info.theGlyph    = 0;
				char_info.index       = l_index;
				char_info.theChar     = text[textloop];
				theText               = text[textloop];
				char_info.angle       = 0;
				term_found            = SVG_FALSE;
				p_renderer ->is_first_char = SVG_TRUE;
				glyphs_to_draw = 0;
				p_renderer->advance_x = 0.0f;
				p_renderer->advance_y = 0.0f;
				p_renderer->kerning   = 0.0f;
				loop= 0;

                                /* calling hb apis to load glyphs and getting metrics*/
                                    if(p_renderer->enableshapingeng == SVG_TRUE)
                                    {
                                        p_renderer->cached_rendrflags   = char_info.flags;
                                        p_renderer->cached_index         = char_info.index;
                                        char_info.textlength                    = textLength[textloop];

                                        error = p_renderer->p_grl_shpeng_fp_table->glyphshaping(
                                                                                (void*)p_renderer->shared->p_shape_eng,
                                                                                (get_char_info_struct*)&char_info,
                                                                                (void*)ctx,
                                                                                ctx->encoding );

                                        if (error != SVG_NO_ERROR)
                                        {
                                            hbGlyphCount = 0;
                                            grlftSetError(ctx, error);
                                        }
                                        else
                                        {
                                            hbGlyphCount =  p_renderer->p_grl_shpeng_fp_table->GetGlyphCount(
                                                                                        (void*)p_renderer->shared->p_shape_eng);
                                        }

                                        if(!hbGlyphCount)
                                        {
                                            term_found = SVG_TRUE;
                                        }
                                   }

				//(char_info.theChar < &theText[textLength[textloop]])
				
				while(SVG_FALSE == term_found)
				{
					/*get the FT bitmap and store it to the corresponding "texture" */
					error = local_get_bitmap( ctx, &char_info, loop, SVG_TRUE, &add_new_glyph,NULL);

					if (error != SVG_NO_ERROR)
					{
						if (error != SVG_NULL_CHARACTER)
						{
							grlftSetError(ctx, error);
							if(error == TOO_MANY_GLYPHS_FOR_CACHE)
							{
								term_found = SVG_TRUE;
								textNumber = textloop;
							}
						}
						else
						{
							if(p_renderer->enableshapingeng == SVG_FALSE)
							{
								if(char_info.decodedChar == 0)
								{
									term_found = SVG_TRUE;
								}
							}
						}
					}
					else
					{
						/* store information to application space */
						texturebox[textloop][loop].top    = p_renderer->shared->cache_current->texture_source_y;
						texturebox[textloop][loop].left   = p_renderer->shared->cache_current->texture_source_x;
						texturebox[textloop][loop].width  = p_renderer->shared->cache_current->texture_width;
						texturebox[textloop][loop].height = p_renderer->shared->cache_current->texture_height;
				
						glyphbox[textloop][loop].top      = p_renderer->offset_y;
						glyphbox[textloop][loop].left     = p_renderer->offset_x;
						glyphbox[textloop][loop].width    = p_renderer->shared->cache_current->width;
						glyphbox[textloop][loop].height   = p_renderer->shared->cache_current->height;
						glyphs_to_draw++;
						/*offset_y is relative value but we need a absolute gap to the baseline*/
						absl_offset_y += p_renderer->offset_y;
						if( absl_offset_y < max_gap[textloop] )
						{
							max_gap[textloop] = absl_offset_y;
						}
					 }

					 loop++;
					 
					 if(p_renderer->enableshapingeng == SVG_FALSE)
					 {
						 char_info.theChar = &char_info.theChar[1];
						 if(!(char_info.theChar < &theText[textLength[textloop]]))
						 {
						 	term_found = SVG_TRUE;
						 }
					 }
					 else
					 {
                    	if(loop == hbGlyphCount)
                    	{
							term_found = SVG_TRUE;	
                    	}
	                 } 
				}
				glyphNumber[textloop] = glyphs_to_draw;
				absl_offset_y = 0;

				if(p_renderer->enableshapingeng == SVG_TRUE)
				{
					/* destory hb buffer object*/
					p_renderer->p_grl_shpeng_fp_table->DestroyBuffObjects(
										(void*)p_renderer->shared->p_shape_eng);
                                    /* Reset shapingeng flag*/
                                    p_renderer->enableshapingeng = SVG_FALSE;

				}
				
			}
			if ( (l_index == 0) && (add_new_glyph))
            {
                is_update_strk = SVG_TRUE;
            }
            if ( (l_index == 1) && (add_new_glyph))
            {
                is_update_fill = SVG_TRUE;
            }
		}
		reset_in_use_glyph_flag(p_renderer);
		
		for(l_index = start; l_index < end; l_index++)
		{   	
			if( l_index == 0 )
			{
				glyphbox      = strokedGlyphBox;
				texturebox    = strokedTextureBox;
				
				#ifdef HAS_OPENVG
					if(SVG_USE_OPENVG == p_renderer->drawingAPI)
					{
						*strokedTexture = (SVGUint32)p_renderer->shared->vgWallpaperStrk;
					}
				#endif                                    
				#ifdef HAS_OPENGLES
					if(SVG_USE_OPENGLES_1 == p_renderer->drawingAPI)
					{      
						*strokedTexture = (SVGUint32)p_renderer->shared->glWallpaperStrk;
					#ifdef HAS_OPENGLES2
						if ( ((ctx->settings & SVG_FONT_MIPMAPPING) == SVG_FONT_MIPMAPPING) &&
								(is_update_strk) )
						{       
							glBindTexture(GL_TEXTURE_2D,*strokedTexture);
							glGenerateMipmap(GL_TEXTURE_2D);
						}
					#endif
					}
				#endif 
			}
			else
			{
				glyphbox      = filledGlyphBox;
				texturebox    = filledTextureBox;
				
				#ifdef HAS_OPENVG
					if(SVG_USE_OPENVG == p_renderer->drawingAPI)
					{
						*filledTexture = (SVGUint32)p_renderer->shared->vgWallpaperFill;

					}
				#endif                                    
				#ifdef HAS_OPENGLES
					if(SVG_USE_OPENGLES_1 == p_renderer->drawingAPI)
					{      
						*filledTexture = (SVGUint32)p_renderer->shared->glWallpaperFill;
					#ifdef HAS_OPENGLES2	
						if (((ctx->settings & SVG_FONT_MIPMAPPING) == SVG_FONT_MIPMAPPING) &&
								(is_update_fill))
						{       
							glBindTexture(GL_TEXTURE_2D,*filledTexture);
							glGenerateMipmap(GL_TEXTURE_2D); 
						}
					#endif
					}
				#endif 
			}
			if( (ctx->settings & SVG_FONT_NO_ENCLOSING_SPACES) != 0 )
			{
				for(textloop = 0; textloop < textNumber; textloop++)
				{
					glyphbox[textloop][0].top -= max_gap[textloop];
				}
			}
			else
			{
				for(textloop = 0; textloop < textNumber; textloop++)
				{
					glyphbox[textloop][0].top += p_renderer->descender + (SVGFloat)(p_renderer->stroke_width >> 1);
				}
			}
		}


		ctx->letter_distance = distance;
			
		GRL_free_1D_resource(max_gap);
		if(NULL != p_script_rendr_data)
                GRL_free_1D_resource(p_script_rendr_data);
	}
	else
	{
		grlftSetError(ctx, SVG_OUT_OF_MEMORY);
	    SVG_FNT_F("SVG_OUT_OF_MEMORY IN GRLXXGETFONTBITMAPEXT");
	}

	local_check_parallel_use( ctx->thread_count );
	__sync_sub_and_fetch(&ctx->thread_count, 1);
	if(NULL != ctx->shared)
    {
        GRL_sig_lock(p_renderer->shared->id);
    }

	return textloop;                                             
}                                 

void grlftGetFontBitmap( SVGFontContext* ctx,
                         SVGUint8* text,
                         SVGUint32 textLength,
                         SVGUint32* texture,
                         SVGFloat *width,
                         SVGFloat *height)
{
    SVGFontFreeTypeRenderer*    p_renderer    = (SVGFontFreeTypeRenderer*)
                                                 ctx->p_fontRendererContext;
    SVGPoint                    position      = {0.0f, 0.0f};
    SVGBBox                     bbox          = {0.0f, 0.0f, 0.0f, 0.0f};
    SVGBoolean                  do_paint      = SVG_TRUE;
    SVGUint32                   settings      = ctx->settings;
    SVGFloat                    org_angle     = ctx->angle;
    SVGOriginType               org_base_line = ctx->base_line;
    SVGFloat                    org_scale_x	  = ctx->scale_x;
    SVGFloat                    org_scale_y   = ctx->scale_y;
#ifdef HAS_OPENVG
    VGImage                     VGimage       = VG_INVALID_HANDLE;
#endif    
#ifdef HAS_OPENGLES
    GLuint                      GLimage       = 0;
    SVGUint32                   a_width         = 0;
    SVGUint32                   a_height        = 0;
    SVGUint32                   i             = 0;
    SVGUint8                    *Resource     = NULL;
#endif
	
    *texture = 0;
    if (textLength != 0)
    {
        if(NULL != ctx->shared)
        {
            GRL_wai_lock(p_renderer->shared->id);
        }
        __sync_add_and_fetch(&ctx->thread_count, 1);//atomic incrementation, used for parallel access detection
        local_check_parallel_use( ctx->thread_count );
		
    	/*reset angle, scale and baseline for font bitmap*/
        ctx->angle 		= 0.0f;
        ctx->base_line 	= SVG_ORIGIN_BOTTOM;
        ctx->scale_x	= 1.0f; 
		ctx->scale_y	= 1.0f; 
        
        /* when 2-color is activated just use outline instead, 
           because then only 1 pass is needed */
        if ((ctx->settings & SVG_FONT_2COLOR) != 0)
        {
            ctx->settings &= (0xFFFFFFFF ^ SVG_FONT_2COLOR);
            ctx->settings |= SVG_FONT_OUTLINE;
        }
        local_drawFontWithCursorInfo( ctx,
                                      &position,
                                      &ctx->angle,
                                      text,
                                      textLength,
                                      NULL,
                                      SVG_FALSE,
                                      &bbox);

#ifdef HAS_OPENVG
        if(SVG_USE_OPENVG == p_renderer->drawingAPI)
        {
            /* create image */
            VGimage = vgCreateImage( VG_A_8, 
                                     (SVGInt32)bbox.width, 
                                     (SVGInt32)bbox.height, 
                                     VG_IMAGE_QUALITY_FASTER);
            if (VG_INVALID_HANDLE == VGimage)
            {
                *texture = 0;
                grlftSetError(ctx, VG_TEXTURE_CREATION_FAIL);
        	    SVG_FNT_E("VG_TEXTURE_CREATION_FAIL "
        	    		 "IN GRLXXGETFONTBITMAP");
                do_paint = SVG_FALSE;
            }
            else
            {
                *texture = (SVGUint32)VGimage;                                          
            }
        }
#endif
#ifdef HAS_OPENGLES
        if(SVG_USE_OPENGLES_1 == p_renderer->drawingAPI)
        {
            /* create image */
            glGenTextures(1, &GLimage);
            if (0 == GLimage)
            {
                *texture = 0;
                grlftSetError(ctx, GL_TEXTURE_CREATION_FAIL);
        	    SVG_FNT_E("GL_TEXTURE_CREATION_FAIL "
        	    		 "IN GRLXXGETFONTBITMAP");
                do_paint = SVG_FALSE;
            }
            else
            {
                glBindTexture(GL_TEXTURE_2D, GLimage);
                a_width  = (SVGUint32)bbox.width;
                a_height = (SVGUint32)bbox.height;
                if((a_width &(a_width-1)) != 0)
                {
                    for (i = 1; a_width > i; i <<= 1)
                    {
                    	/* align a_width to power of 2 */
                    }
                    a_width = i;
                }
            
                if((a_height &(a_height-1)) != 0)
                {
                    for (i = 1; a_height > i; i <<= 1)
                    {
                    	/* align a_height to power of 2 */
                    }
                    a_height = i;
                }
                /*get a resource of the image dimension*/
                Resource = (SVGUint8*)GRL_malloc_1D_resource(a_height * a_width);
			    if (NULL != Resource)
			    {
			    	memset(Resource, 0, a_height*a_width);
			    	glTexImage2D(GL_TEXTURE_2D, 
                             0, 
                             GL_ALPHA,
                             (GLsizei)(a_width), 
                             (GLsizei)(a_height), 
                             0, 
                             GL_ALPHA,
                             GL_UNSIGNED_BYTE,
                             Resource);
				}
				else
				{
					glTexImage2D(GL_TEXTURE_2D, 
                             0, 
                             GL_ALPHA,
                             (GLsizei)(a_width), 
                             (GLsizei)(a_height), 
                             0, 
                             GL_ALPHA,
                             GL_UNSIGNED_BYTE,
                             NULL);
	        	    SVG_FNT_W("Wallpaper not cleared ");
				}
                
                glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
                glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    
                glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
                glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
#ifdef HAS_OPENGLES1

        		glTexEnvx(GL_TEXTURE_ENV, 
	                 		    GL_TEXTURE_ENV_MODE, 
                      			GL_REPLACE);
                                        
                if ((ctx->settings & SVG_FONT_MIPMAPPING) == SVG_FONT_MIPMAPPING)
                {                                
					glTexParameteri(GL_TEXTURE_2D, 
									GL_GENERATE_MIPMAP,
									GL_TRUE);                                                                                              
				}                                                                                    
#else
                if ((ctx->settings & SVG_FONT_MIPMAPPING) == SVG_FONT_MIPMAPPING)
                {                                
					glGenerateMipmap(GL_TEXTURE_2D); 
					glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
				}                                                                                    
#endif				
                *texture = (SVGUint32)GLimage;
                
                /*release resource*/
                GRL_free_1D_resource((void*)Resource);
            }
        }
#endif        
        
        if(SVG_TRUE == do_paint)
        {
            /* draw sting onto the newly created image */
            p_renderer->texture = *texture;
            local_drawFontWithCursorInfo( ctx,
                                          &position,
                                          &ctx->angle,
                                          text,
                                          textLength,
                                          NULL,
                                          SVG_FALSE,
                                          NULL);      
            *width  = bbox.width;
            *height = bbox.height;                                          
            p_renderer->texture = 0;
#ifdef HAS_OPENGLES
            if(SVG_USE_OPENGLES_1 == p_renderer->drawingAPI)
            {        	
                glBindTexture(GL_TEXTURE_2D, p_renderer->shared->glWallpaperFill);
            }
#endif        
        }
        /*restore angle, scale and baseline*/
        ctx->angle 		= org_angle;
        ctx->base_line 	= org_base_line;
        ctx->scale_x	= org_scale_x;
		ctx->scale_y	= org_scale_y;
        ctx->settings = settings;

        local_check_parallel_use( ctx->thread_count );
        __sync_sub_and_fetch(&ctx->thread_count, 1);
        if(NULL != ctx->shared)
        {
         GRL_sig_lock(p_renderer->shared->id);
        }
    }
    else
    {
        grlftSetError(ctx, SVG_INVALID_VALUE);
	    SVG_FNT_E("SVG_INVALID_VALUE IN GRLXXGETFONTBITMAP");
    }
}

void grlftFreeFontBitmapMem( SVGFontContext* ctx, SVGUint32 texture)
{
    SVGFontFreeTypeRenderer*    p_renderer   = (SVGFontFreeTypeRenderer*)
                                                ctx->p_fontRendererContext;
    if (texture != 0)
    {
#ifdef HAS_OPENVG
        if(SVG_USE_OPENVG == p_renderer->drawingAPI)
        {
            vgDestroyImage((VGImage)texture);
        }
#endif        
#ifdef HAS_OPENGLES
        if(SVG_USE_OPENGLES_1 == p_renderer->drawingAPI)
        {
            glDeleteTextures(1, (GLuint*)&texture);
        }
#endif                
    }
    else
    {
        grlftSetError(ctx, SVG_POINTER_NULL);
	    SVG_FNT_E("SVG_POINTER_NULL IN GRLXXFREEFONTBITMAPMEM");
    }
}

void grlftDrawFontWithCursorInfo( SVGFontContext* ctx,
                                   SVGFloat positionX,
                                   SVGFloat positionY,
                                   SVGUint8* text,
                                   SVGUint32 textLength,
                                   SVGPoint* cursorPosArray )
{
    SVGPoint position;
    SVGFloat a_angle = ctx->angle;

    if(NULL != ctx->shared)
    {
        GRL_wai_lock( ((SVGFontFreeTypeRenderer*)ctx->p_fontRendererContext)->shared->id);
    }
    __sync_add_and_fetch(&ctx->thread_count, 1);//atomic incrementation, used for parallel access detection
    local_check_parallel_use( ctx->thread_count );

    position.x = (SVGFloat)positionX;
    position.y = (SVGFloat)positionY;
    local_drawFontWithCursorInfo(ctx, 
                                 &position, 
                                 &a_angle, 
                                 text, 
                                 textLength, 
                                 cursorPosArray, 
                                 SVG_FALSE,
                                 NULL);
    local_check_parallel_use( ctx->thread_count );
    __sync_sub_and_fetch(&ctx->thread_count, 1);
	
    if(NULL != ctx->shared)
    {
     GRL_sig_lock( ((SVGFontFreeTypeRenderer*)ctx->p_fontRendererContext)->shared->id);
    }
}

void grlftGetFontBBox( SVGFontContext* ctx,
                       SVGUint8* text,
                       SVGUint32 textLength,
                       SVGBBox* textBBox)
{
    SVGPoint position = {0.0f, 0.0f};
    SVGFloat a_angle = ctx->angle;
    SVGUint32 settings = ctx->settings;

    if(NULL != ctx->shared)
    {
        GRL_wai_lock( ((SVGFontFreeTypeRenderer*)ctx->p_fontRendererContext)->shared->id);
    }
    __sync_add_and_fetch(&ctx->thread_count, 1);//atomic incrementation, used for parallel access detection
	local_check_parallel_use( ctx->thread_count );
	
    /* when 2-color is activated just use outline instead, 
       because then only 1 pass is needed */
    if ((ctx->settings & SVG_FONT_2COLOR) == SVG_FONT_2COLOR)
    {
        ctx->settings &= (0xFFFFFFFF ^ SVG_FONT_2COLOR);
        ctx->settings |= SVG_FONT_OUTLINE;
    }
    /* reset values to ensure correct run */
    textBBox->width  = 0;
    textBBox->height = 0;
    textBBox->left   = 0;
    textBBox->top    = 0;
    local_drawFontWithCursorInfo(ctx, 
                                 &position, 
                                 &a_angle, 
                                 text, 
                                 textLength, 
                                 NULL, 
                                 SVG_FALSE,
                                 textBBox);
    ctx->settings = settings;
    local_check_parallel_use( ctx->thread_count );
    __sync_sub_and_fetch(&ctx->thread_count, 1);

    if(NULL != ctx->shared)
    {
     GRL_sig_lock( ((SVGFontFreeTypeRenderer*)ctx->p_fontRendererContext)->shared->id);
    }
}


SVGUint32 grlftGetFontMaxChars( SVGFontContext* ctx,
                                SVGFloat max_space_x,
                                SVGFloat max_space_y,
                                SVGUint8* text,
                                SVGUint32 textLength)
{
    SVGBBox  textBBox = {0.0f, 0.0f, 0.0f, 0.0f};
    SVGPoint position = {0.0f, 0.0f};
    SVGFloat a_angle  = ctx->angle;
    SVGUint32 maxchars = 0;
    SVGUint32 settings = ctx->settings;

    if(NULL != ctx->shared)
    {
        GRL_wai_lock(((SVGFontFreeTypeRenderer*)ctx->p_fontRendererContext)->shared->id);
    }
    __sync_add_and_fetch(&ctx->thread_count, 1);//atomic incrementation, used for parallel access detection
    local_check_parallel_use( ctx->thread_count );
    /* when 2-color is activated just use outline instead, 
       because then only 1 pass is needed */
    if ((ctx->settings & SVG_FONT_2COLOR) == SVG_FONT_2COLOR)
    {
        ctx->settings &= (0xFFFFFFFF ^ SVG_FONT_2COLOR);
        ctx->settings |= SVG_FONT_OUTLINE;
    }

    textBBox.width    = max_space_x;
    textBBox.height   = max_space_y;

    maxchars = local_drawFontWithCursorInfo( ctx, 
                                             &position, 
                                             &a_angle, 
                                             text, 
                                             textLength, 
                                             NULL, 
                                             SVG_FALSE,
                                             &textBBox);
    ctx->settings = settings;
    local_check_parallel_use( ctx->thread_count );
    __sync_sub_and_fetch(&ctx->thread_count, 1);
    if(NULL != ctx->shared)
    {
        GRL_sig_lock(((SVGFontFreeTypeRenderer*)ctx->p_fontRendererContext)->shared->id);
    }

    return maxchars;                                             
}


void  grlftGetFontInformation ( SVGFontContext *ctx,
                                SVGFontInfo *info )
{
    SVGFontFreeTypeRenderer*    p_renderer =
        ((SVGFontFreeTypeRenderer*)ctx->p_fontRendererContext);

    if (info == NULL)
    {
        grlftSetError (ctx, SVG_POINTER_NULL);
    }
    else
    {
        info->ascender              = p_renderer->ascender;
        info->descender             = p_renderer->descender;
        info->underline_position    = p_renderer->underline_position;
        info->underline_thickness   = p_renderer->underline_thickness;
        info->encoding              = ctx->encoding;
        info->font_size             = (SVGUint8)p_renderer->font_size;
        info->font                  = p_renderer->shared->p_svgFont;
        info->wallpaper_width_height= 0;
        if (SVG_USE_OPENVG == p_renderer->drawingAPI)
        {
#ifdef HAS_OPENVG
        	info->wallpaper_width_height = p_renderer->shared->vgWallpaperWidth;
#endif
        }
        else
        {
#ifdef HAS_OPENGLES
        	info->wallpaper_width_height = p_renderer->shared->glWallpaperWidth;
#endif
        }
    }
}

void grlftDrawFontStringExt(SVGFontContext* ctx, 
                            SVGPoint* position, 
                            SVGFloat* angle, 
                            SVGUint8* glyphs,
                            SVGUint32 glyphLength)
{
    SVGFloat       langle    = ctx->angle;
    SVGUint32      settings  = ctx->settings;
    SVGOriginType  base_line = ctx->base_line;

    if(NULL != ctx->shared)
    {
        GRL_wai_lock(((SVGFontFreeTypeRenderer*)ctx->p_fontRendererContext)->shared->id);
    }
    __sync_add_and_fetch(&ctx->thread_count, 1);//atomic incrementation, used for parallel access detection
    local_check_parallel_use( ctx->thread_count );


    /* reset vertical and no_enclosing_spaces setting */
    ctx->settings &= (0xFFFFFFFF ^ (SVG_FONT_VERTICAL | SVG_FONT_NO_ENCLOSING_SPACES | SVG_FONT_MIPMAPPING));
    ctx->angle = angle[0];
    /* reset origin to BASELINE */
    ctx->base_line = SVG_ORIGIN_BASELINE;

    local_drawFontWithCursorInfo(ctx, 
                                 position, 
                                 angle, 
                                 glyphs, 
                                 glyphLength, 
                                 NULL, 
                                 SVG_TRUE,
                                 NULL);
    /*restore settings after drawing*/
    ctx->angle     = langle;
    ctx->settings  = settings;
    ctx->base_line = base_line;

    local_check_parallel_use( ctx->thread_count );
    __sync_sub_and_fetch(&ctx->thread_count, 1);
	
    if(NULL != ctx->shared)
    {
     GRL_sig_lock( ((SVGFontFreeTypeRenderer*)ctx->p_fontRendererContext)->shared->id);
    }
}

static void reset_in_use_glyph_flag(SVGFontFreeTypeRenderer* p_renderer)
{
    grl_internal_cache_struct *cache_current = p_renderer->shared->cache_strk;
	while((cache_current != NULL) && (cache_current->is_in_use == SVG_TRUE))
	{
		cache_current->is_in_use = SVG_FALSE;
		cache_current = cache_current->next;
	}
	
	cache_current = p_renderer->shared->cache_fill;
	while((cache_current != NULL) && (cache_current->is_in_use == SVG_TRUE))
	{
		cache_current->is_in_use = SVG_FALSE;
		cache_current = cache_current->next;
	}
}

static void local_clear_cache(SVGFontFreeTypeRenderer* p_renderer, SVGBoolean filled)
{
	grl_internal_cache_struct  *cache_search      = NULL;
    grl_internal_cache_struct  *prev_cache_search = NULL;
	if(filled)
		cache_search = p_renderer->shared->cache_fill;
	else
		cache_search = p_renderer->shared->cache_strk;
	
	while(NULL != cache_search)
	{
	#ifdef HAS_OPENVG
		if(SVG_USE_OPENVG == p_renderer->drawingAPI)
		{
			if( 0 != cache_search->VGimage)
			{
				vgDestroyImage(cache_search->VGimage);
			}
		}
	#endif
		if (cache_search->glyph_done == SVG_FALSE)
		{
			FT_Done_Glyph(cache_search->ft_glyph);	
		}
		prev_cache_search = cache_search;                
		cache_search = cache_search->next;                
		GRL_free_1D_resource((void*)prev_cache_search);                
	}
	
}


static void local_del_strk_glyphs( SVGFontContext* ctx )
{
	SVGError 				   ret 				  = SVG_NO_ERROR;
	SVGFontFreeTypeRenderer    *p_renderer		  = (SVGFontFreeTypeRenderer*)
                                                     ctx->p_fontRendererContext;
	UTIL_hash_elem_t		   entry_to_find	  = {NULL, 0, NULL};
    grl_internal_cache_struct *cache_current      = p_renderer->shared->cache_strk;
	grl_internal_cache_struct *cache_tmp	      = NULL;
	SVGBoolean				   cleanStrkWall	  = SVG_FALSE;
	textureRow				  *rowStrk       	  = NULL;
    textureRow				  *cur_row	    	  = NULL;
    textureCol				  *col      		  = NULL;
    textureCol				  *cur_col   		  = NULL;
	
    p_renderer->shared->cache = &p_renderer->shared->cache_strk;
	p_renderer->shared->cache_count = &p_renderer->shared->cache_count_strk;
	p_renderer->shared->hash_tbl_glyphs = p_renderer->shared->hash_tbl_strk;

	while((NULL != cache_current) && (ret == SVG_NO_ERROR))
	{
		if(	cache_current->cached_bitmap == SVG_TRUE )
		{
			cleanStrkWall = SVG_TRUE;
			
			entry_to_find.key = cache_current;
			entry_to_find.key_len = sizeof(SVGUint32)*3;
			entry_to_find.value = cache_current;

			/* remove the glyph from the hash table */
			ret = UTIL_rem_hsh(p_renderer->shared->hash_tbl_glyphs,&entry_to_find);
			if( ret == SVG_NO_ERROR)
			{
#ifdef HAS_OPENVG
				if(SVG_USE_OPENVG == p_renderer->drawingAPI)
				{
					if( 0 != cache_current->VGimage)
					{
						vgDestroyImage(cache_current->VGimage);
					}
				}
#endif
				if (cache_current->glyph_done != SVG_TRUE)
				{
					FT_Done_Glyph(cache_current->ft_glyph);
				}
				
				/* remove resource */
				(*p_renderer->shared->cache_count)--;
				if( cache_current->next == NULL)
				{
					if(cache_current->prev != NULL)
					{
						cache_current->prev->next = cache_current->next;
					}
					else
					{/*last element in cache(prev and next are NULL) */
						*p_renderer->shared->cache = NULL;
					}
					GRL_free_1D_resource((void*)cache_current);
					cache_current = NULL;
				}
				else
				{
					if(cache_current->prev != NULL)
					{
						cache_current->prev->next = cache_current->next;
						cache_current->next->prev = cache_current->prev;
					}
					else
					{
						*p_renderer->shared->cache = cache_current->next;
						(*p_renderer->shared->cache)->prev = NULL;
					}
					cache_tmp = cache_current;
					
					cache_current = cache_current->next;
					GRL_free_1D_resource((void*)cache_tmp);
					
				}
			}
			else
			{
				/*error by removing hash value, hash table corrupted???*/
				grlftSetError(ctx, HASHENTRY_REMOVE_ERROR);
			    SVG_FNT_E("Hash element could not be removed "
			    		"it's can leads to cache inconsistency and memory leaks");
			}
		}
		else
		{
			cache_current = cache_current->next;
		}
	}
	p_renderer->shared->cache_current = p_renderer->shared->cache_strk;
	if( cleanStrkWall == SVG_TRUE)
	{
		/*remove stroked wallpaper*/
#ifdef HAS_OPENVG
		if(SVG_USE_OPENVG == p_renderer->drawingAPI)
		{
			rowStrk = p_renderer->shared->vgRowStrk;
			p_renderer->shared->vgRowStrk = NULL;
			
			if( 0 != p_renderer->shared->vgWallpaperStrk)
			{
				vgDestroyImage(p_renderer->shared->vgWallpaperStrk);
				p_renderer->shared->vgWallpaperStrk = 0;
			}
		}
#endif                
#ifdef HAS_OPENGLES
		if(SVG_USE_OPENGLES_1 == p_renderer->drawingAPI)
		{
			rowStrk = p_renderer->shared->glRowStrk;
			p_renderer->shared->glRowStrk = NULL;
			
			if( 0 != p_renderer->shared->glWallpaperStrk)
			{
				glDeleteTextures(1, (GLuint*)&p_renderer->shared->glWallpaperStrk);
				p_renderer->shared->glWallpaperStrk = 0;
			}
			
		}
#endif                	
		while(NULL != rowStrk)
		{
			col = rowStrk->col;
			while(NULL != col)
			{
				cur_col = col;
				col = col->next;
				GRL_free_1D_resource((void*)cur_col);
			}
			cur_row = rowStrk;
			rowStrk = rowStrk->next;
			GRL_free_1D_resource((void*)cur_row);
		}
	/*creates neccessary wallpaper in this case only stroked will be created*/
		local_create_wallpapers( ctx, p_renderer->font_size);
	}
}

static void local_create_wallpapers( SVGFontContext* ctx  , SVGUint32 fontSize)
{  

	SVGFontFreeTypeRenderer*    p_renderer        = (SVGFontFreeTypeRenderer*)
                                                     ctx->p_fontRendererContext;
	SVGUint32                   glyphs_per_row    = (SVGUint32)sqrt(p_renderer->cachable_glyphs);
    SVGUint32                   width_height      = 0;
	SVGUint32					strk_flag		  = 0;
	SVGUint32					fill_flag		  = 0;
#ifdef HAS_OPENGLES
    SVGUint8*                   Resource          = NULL;
    SVGUint32                   i                 = 0;
#endif  

#ifdef HAS_OPENVG
	if(SVG_USE_OPENVG == p_renderer->drawingAPI)
	{	/*create only if null and outline or 2color are set*/
		if( p_renderer->shared->vgWallpaperWidth > 0)
			width_height = p_renderer->shared->vgWallpaperWidth;
		else
			width_height = (glyphs_per_row + MORE_CHARS_THEN_NEEDED ) * (SVGUint32)((SVGFloat)(fontSize) * FILLED_PIXEL_RATIO);
			
		if( (0 == p_renderer->shared->vgWallpaperStrk) &&
			( ((ctx->settings & SVG_FONT_OUTLINE) != 0) || ((ctx->settings & SVG_FONT_2COLOR) != 0)) )
        {
			p_renderer->shared->vgWallpaperWidth = width_height;
			p_renderer->shared->vgWallpaperHeight = width_height;
			p_renderer->shared->vgWallpaperStrk = vgCreateImage( VG_A_8, 
															 (SVGInt32)width_height, 
															 (SVGInt32)width_height, 
															  VG_IMAGE_QUALITY_FASTER);
			strk_flag = 1;
		    SVG_FNT_D("Create wallpaper for font glpyhs"
		    		" with dimension %d x %d"
		    		,width_height,width_height);
		}
		/*create only if null and */
		if( (0 == p_renderer->shared->vgWallpaperFill) &&
		  !( ((ctx->settings & SVG_FONT_OUTLINE) != 0)  && ((ctx->settings & SVG_FONT_2COLOR) == 0)) )
        {
			p_renderer->shared->vgWallpaperWidth  = width_height;
			p_renderer->shared->vgWallpaperHeight = width_height;
			p_renderer->shared->vgWallpaperFill = vgCreateImage( VG_A_8, 
															 (SVGInt32)width_height, 
															 (SVGInt32)width_height, 
															  VG_IMAGE_QUALITY_FASTER);														
			fill_flag = 1;
		    SVG_FNT_D("Create wallpaper for font glpyhs"
		    		" with dimension %d x %d"
		    		,width_height,width_height);
		}
		if(  ((0 == p_renderer->shared->vgWallpaperStrk) && strk_flag )  || 
			  ((0 == p_renderer->shared->vgWallpaperFill) && fill_flag ))
		{
		    SVG_FNT_E("VG_TEXTURE_CREATION_FAIL IN GRLXXSETFONTSIZE");
			grlftSetError(ctx, VG_TEXTURE_CREATION_FAIL);
		}                                                                     
	}
#endif                                                                                             
#ifdef HAS_OPENGLES
	if(SVG_USE_OPENGLES_1 == p_renderer->drawingAPI)
	{
		if(p_renderer->shared->glWallpaperWidth > 0)
			width_height = p_renderer->shared->glWallpaperWidth;
		else
			width_height = (glyphs_per_row + MORE_CHARS_THEN_NEEDED ) * (SVGUint32)((SVGFloat)(fontSize) * FILLED_PIXEL_RATIO);

		if((width_height &(width_height-1)) != 0)
		{
			for (i = 1; width_height > i; i <<= 1)
			{
				/* align width to power of 2 */
			}
			width_height = i;
		}


		/* limit texture size to its maximum size */
		if (width_height > p_renderer->max_texture_size)
		{
			width_height = p_renderer->max_texture_size;

			if( fontSize !=0 )
			{
				p_renderer->cachable_glyphs = ((SVGUint32)((width_height / FILLED_PIXEL_RATIO)/ ((SVGFloat)(fontSize)))- MORE_CHARS_THEN_NEEDED);
			}
			else
			{
				p_renderer->cachable_glyphs = 0;
			}

			p_renderer->cachable_glyphs *= p_renderer->cachable_glyphs;
		    SVG_FNT_E("limiting texture size to %d and "
		    		"cachable glyphs to %d due to HW limitation"
		    		,width_height,p_renderer->cachable_glyphs);
		}

		if( (0 == p_renderer->shared->glWallpaperStrk) &&
			( ((ctx->settings & SVG_FONT_OUTLINE) != 0) || ((ctx->settings & SVG_FONT_2COLOR) != 0)) )
        {
			p_renderer->shared->glWallpaperWidth  = width_height;
			p_renderer->shared->glWallpaperHeight = width_height;
			glGenTextures(1, &p_renderer->shared->glWallpaperStrk);
			
			strk_flag = 1;
		    SVG_FNT_D("Create wallpaper for font glpyhs "
		    		"with dimension %d x %d"
		    		,width_height,width_height);
		}
		
		if( (0 == p_renderer->shared->glWallpaperFill) &&
		  !( ((ctx->settings & SVG_FONT_OUTLINE) != 0)  && ((ctx->settings & SVG_FONT_2COLOR) == 0)) )
        {
			p_renderer->shared->glWallpaperWidth  = width_height;
			p_renderer->shared->glWallpaperHeight = width_height;
			glGenTextures(1, &p_renderer->shared->glWallpaperFill);
			
			fill_flag = 1;
		    SVG_FNT_D("Create wallpaper for font glpyhs "
		    		"with dimension %d x %d"
		    		,width_height,width_height);
		}
		
		if(  ((0 == p_renderer->shared->glWallpaperStrk) && strk_flag )  || 
			  ((0 == p_renderer->shared->glWallpaperFill) && fill_flag ))
		{
		    SVG_FNT_E("GL_TEXTURE_CREATION_FAIL IN GRLXXSETFONTSIZE");
			grlftSetError(ctx, GL_TEXTURE_CREATION_FAIL);
		}                                                                     
		else
		{	
						
			Resource = (SVGUint8*)GRL_malloc_1D_resource(width_height * width_height);
			/*two passed to clear both wallpapers filled and stroked*/
			for( i = 0; i < (fill_flag+strk_flag); i++)
			{
				if( ((i+fill_flag) < 2) && (fill_flag != 0) )
				{
					glBindTexture(GL_TEXTURE_2D, p_renderer->shared->glWallpaperFill);
				}
				else
				{
					glBindTexture(GL_TEXTURE_2D, p_renderer->shared->glWallpaperStrk);				
				}
				if (NULL != Resource)
				{
					memset(Resource, 0, width_height * width_height);
					glTexImage2D(GL_TEXTURE_2D, 
							 0, 
							 GL_ALPHA,
							 (GLsizei)(width_height), 
							 (GLsizei)(width_height), 
							 0, 
							 GL_ALPHA,
							 GL_UNSIGNED_BYTE,
							 Resource);
				}
				else
				{
					glTexImage2D(GL_TEXTURE_2D, 
							 0, 
							 GL_ALPHA,
							 (GLsizei)(width_height), 
							 (GLsizei)(width_height), 
							 0, 
							 GL_ALPHA,
							 GL_UNSIGNED_BYTE,
							 NULL);
				    SVG_FNT_W("Wallpaper not cleared");
				}
			/*NOTE: stred wallpaper is set as current texture*/
				glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR /*GL_NEAREST*/);
				glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR /*GL_NEAREST*/);

				glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
				glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
                if ( (ctx->settings & SVG_FONT_MIPMAPPING) == SVG_FONT_MIPMAPPING)
                {
#ifdef HAS_OPENGLES1
                    glTexEnvx(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);

                    glTexParameteri(GL_TEXTURE_2D,
                                    GL_GENERATE_MIPMAP,
                                    GL_TRUE);
#else
                    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
#endif
                }
			}			
			GRL_free_1D_resource((void*)Resource);
		}
	}
#endif   	
}

static void local_free_wallpapers( SVGFontContext* ctx  )
{

	SVGFontFreeTypeRenderer*    p_renderer        = (SVGFontFreeTypeRenderer*)
													ctx->p_fontRendererContext;
	textureRow*   rowFill       = NULL;
	textureRow*   rowStrk       = NULL;
    textureRow*   cur_row	    = NULL;
    textureCol*   col      		= NULL;
    textureCol*   cur_col   	= NULL;
	
#ifdef HAS_OPENVG
	if(SVG_USE_OPENVG == p_renderer->drawingAPI)
	{
		rowFill = p_renderer->shared->vgRowFill;
		p_renderer->shared->vgRowFill = NULL;
		rowStrk = p_renderer->shared->vgRowStrk;
		p_renderer->shared->vgRowStrk = NULL;
		
		if( 0 != p_renderer->shared->vgWallpaperFill)
		{
			vgDestroyImage(p_renderer->shared->vgWallpaperFill);
			p_renderer->shared->vgWallpaperFill = 0;
		    SVG_FNT_D("Delete wallpaper for font glpyhs "
		    		"with dimension %d x %d"
		    		,p_renderer->shared->vgWallpaperWidth
		    		,p_renderer->shared->vgWallpaperWidth);
		}
		if( 0 != p_renderer->shared->vgWallpaperStrk)
		{
			vgDestroyImage(p_renderer->shared->vgWallpaperStrk);
			p_renderer->shared->vgWallpaperStrk = 0;
		    SVG_FNT_D("Delete wallpaper for font glpyhs "
		    		"with dimension %d x %d"
		    		,p_renderer->shared->vgWallpaperWidth
		    		,p_renderer->shared->vgWallpaperWidth);
		}
		p_renderer->shared->vgWallpaperWidth = 0;
		p_renderer->shared->vgWallpaperHeight = 0;
	}
#endif                
#ifdef HAS_OPENGLES
	if(SVG_USE_OPENGLES_1 == p_renderer->drawingAPI)
	{
		rowFill = p_renderer->shared->glRowFill;
		p_renderer->shared->glRowFill = NULL;
		rowStrk = p_renderer->shared->glRowStrk;
		p_renderer->shared->glRowStrk = NULL;
		
		if( 0 != p_renderer->shared->glWallpaperStrk)
		{
			glDeleteTextures(1, (GLuint*)&p_renderer->shared->glWallpaperStrk);
			p_renderer->shared->glWallpaperStrk = 0;
		    SVG_FNT_D("Delete wallpaper for font glpyhs "
		    		"with dimension %d x %d"
		    		,p_renderer->shared->glWallpaperWidth
		    		,p_renderer->shared->glWallpaperWidth);
		}
		if( 0 != p_renderer->shared->glWallpaperFill)
		{
			glDeleteTextures(1, (GLuint*)&p_renderer->shared->glWallpaperFill);
			p_renderer->shared->glWallpaperFill = 0;
		    SVG_FNT_D("Delete wallpaper for font glpyhs "
		    		"with dimension %d x %d"
		    		,p_renderer->shared->glWallpaperWidth
		    		,p_renderer->shared->glWallpaperWidth);
		}
		p_renderer->shared->glWallpaperWidth = 0;
		p_renderer->shared->glWallpaperHeight = 0;
	}
#endif                
	while(NULL != rowFill)
	{
		col = rowFill->col;
		while(NULL != col)
		{
			cur_col = col;
			col = col->next;
			GRL_free_1D_resource((void*)cur_col);
		}
		cur_row = rowFill;
		rowFill = rowFill->next;
		GRL_free_1D_resource((void*)cur_row);
	}
	while(NULL != rowStrk)
	{
		col = rowStrk->col;
		while(NULL != col)
		{
			cur_col = col;
			col = col->next;
			GRL_free_1D_resource((void*)cur_col);
		}
		cur_row = rowStrk;
		rowStrk = rowStrk->next;
		GRL_free_1D_resource((void*)cur_row);
	}
}

static SVGBoolean  local_set_font( SVGFontContext* ctx,
                            SVGFont* font,
                            SVGEncoding charMap,
                            SVGFontFreeTypeRenderer* p_renderer )
{
    SVGBoolean          ret             = SVG_TRUE;
    SVGUint8            encoding_id     = 1;    /* for UNI-code */
    FT_CharMap          charmap         = NULL;
    SVGUint8            n               = 0;

    /**
     * Select encoding style to select the correct
     * character map.
     */
    switch (charMap)
    {
        case SVG_ENCODING_SJIS       :
            encoding_id = 2;
            break;
        case SVG_ENCODING_ASCII      :
        case SVG_ENCODING_UTF8       :
            encoding_id = 1;
            break;
        default                      :
            encoding_id = 1;
            break;
    }

    /**
     * Search a character map that supported the
     * encoding style and select the font.
     *
     * Note: Only windows fonts are supported.
     *       (platform id 3)
     */
    if(p_renderer->shared->face->num_charmaps > 1)
    {
        for(n = 0; n < p_renderer->shared->face->num_charmaps; n++)
        {
            charmap = p_renderer->shared->face->charmaps[n];
            if ((charmap->platform_id == 3) &&
                (charmap->encoding_id == encoding_id))
            {
                if (FT_Set_Charmap(p_renderer->shared->face,
                                   charmap) == 0)
                {
                    if (NULL == ctx->shared)
                    {
                        ctx->validFont = SVG_TRUE;
                    }
                    else
                    {
                        ctx->shared->validFont = SVG_TRUE;
                    }
                    ctx->charMapIndex = n;
                    p_renderer->shared->p_svgFont =
                        GRL_get_font_id(ctx, font);
                }
                else
                {
                    ret = SVG_FALSE;
                }
            }
        }
    }
    else
    {
        if (FT_Set_Charmap(p_renderer->shared->face,
                p_renderer->shared->face->charmaps[0]) == 0)
        {
            if (NULL == ctx->shared)
            {
                ctx->validFont = SVG_TRUE;
            }
            else
            {
                ctx->shared->validFont = SVG_TRUE;
            }
            ctx->charMapIndex = 0;
            p_renderer->shared->p_svgFont = GRL_get_font_id(ctx, font);
        }
        else
        {
            ret = SVG_FALSE;
        }
    }

    return ret;
}

/* calculates the maximum boundary of the sequentially given char-sizes */
static void local_calc_maximum_extension( SVGFontContext* ctx,
                                   SVGBBox  *textBBox,
                                   SVGPoint *position,
                                   SVGBoolean first_or_last_char)
{
    SVGFontFreeTypeRenderer   *p_renderer         = (SVGFontFreeTypeRenderer*)
                                                    ctx->p_fontRendererContext;
    SVGFloat                  x_min               = textBBox->left;
    SVGFloat                  y_max               = textBBox->top;
    SVGFloat                  x_max               = x_min + textBBox->width;
    SVGFloat                  y_min               = y_max - textBBox->height;
    SVGFloat                  x_min_new           = 0.0;
    SVGFloat                  y_max_new           = 0.0;
    SVGFloat                  x_max_new           = 0.0;
    SVGFloat                  y_min_new           = 0.0;
    grl_internal_cache_struct *cache_current      = p_renderer->shared->cache_current;
    SVGFloat                  top                 = cache_current->top;
    SVGFloat                  height              = cache_current->height;
    SVGFloat                  width               = cache_current->width;
    SVGFloat                  sin_angle           = p_renderer->sin_angle;
    SVGFloat                  cos_angle           = p_renderer->cos_angle;            
    /* for "left" position of bounding box take always the left and kerning value */
    SVGFloat                  left                = cache_current->left + 
                                                    p_renderer->kerning;
    /* for calculating next position take always the advance , the kerning and 
       the letter distance value */
    SVGFloat                  advance             = cache_current->advance; 
    SVGPoint                  p1                  = {0.0f, 0.0f};
    SVGPoint                  p2                  = {0.0f, 0.0f};
    SVGPoint                  p3                  = {0.0f, 0.0f};
    SVGPoint                  p4                  = {0.0f, 0.0f};
    SVGFloat                  scale_x             = p_renderer->scale_x;                                     
    SVGFloat                  scale_y             = p_renderer->scale_y;                                     

//    printf("bbox: x %.1f, y %.1f, w %.1f, h %.1f\n", x_min, x_max, textBBox->width, textBBox->height);
//    printf("char: left %.1f, top %.1f, w %.1f, h %.1f, adv %.1f\n", left, top, width, height, advance);
    if(SVG_FALSE == p_renderer->is_vertical)
    {
        /* when the enclosing spaces should be taken into account 
           (NO_ENCLOSING_SPACES NOT set) take the complete height for calculation */
        if((ctx->settings & SVG_FONT_NO_ENCLOSING_SPACES) == 0)
        {
            /* ascender and descender does not take stroke width into account->
               so do it "manually" */
            top    = p_renderer->ascender;
            height = top + p_renderer->descender + p_renderer->stroke_width + 1;
            top += (p_renderer->stroke_width >> 1);

            width = advance - cache_current->left;
            if(p_renderer->stroke_width != (SVGUint32)0)
            {
                /*In case bbox->left is negative then we will face a problem with
                 * numeric string display*/
                /*Add the 1/2 of stroke width, as the advance does not include it.
                 If not added it results in lesser bbox size as width+left
                 exceeds advance*/
                /*Add an additional 1/2 of stroke width in order to have equal width for
                    numeric strings of equal length*/
                /*Add additional two pixels because the rendered alpha map from freetype
                 * extends an additional pixel to right/left for some glyphs*/
				width += (SVGFloat)p_renderer->stroke_width + (SVGFloat)2.0;

				if(		(x_min < (SVGFloat)0.0)
					&&  ((-x_min) <= (SVGFloat)
									((p_renderer->stroke_width >> (SVGUint32)1)
									 + (SVGUint32)1)))
				{
					/*negative bbox->left is treated as an additional width. This
					 statement plays a key role in giving equal width for numeric
					 strings of equal length*/
					width += x_min;
				}
            }
        }
        else
        {
            /* when enclosing spaces should not be considered (SVG_NO_ENCLOSING_SPACES enabled)
              remove the left side bearing of the first glyph */
            if((SVG_TRUE == first_or_last_char) &&
               (textBBox->width == 0))
            {
                advance -= cache_current->left;
                left = 0;
            }            
            /* no need to add stroke width because it is in the "normal" width and height */
            if(0 == width)
            {
                /* special case for non-drawable glyphs e.g. whitespace(has no width, but an advance) */
                width = cache_current->advance;
            }
        }
    }
    else
    {
        /* when the enclosing spaces should be taken into account 
           (NO_ENCLOSING_SPACES NOT set) take the complete height for calculation */
        if((ctx->settings & SVG_FONT_NO_ENCLOSING_SPACES) == 0)
        {
            height = -cache_current->advance;
            width  = p_renderer->right - p_renderer->left;
            left   = -width  * 0.5f;
            top    = height * 0.5f;
            if((position->x == 0) && (position->y == 0))
            {
                position->x = height * sin_angle;
                position->y = height * cos_angle;
                p_renderer->extra_ver = 0.0f;
            }
        }
        else
        {
            if(SVG_TRUE == first_or_last_char)
            {
                if(0 == height)                
                {
                    /* special case for non-drawable glyphs e.g. whitespace(has no width, but an advance) */
                    height = -(cache_current->advance);
                }
                else
                {
                    height = -(cache_current->advance - cache_current->top);
                }
            }
            top = height * 0.5f;
            if((position->x == 0) && (position->y == 0))
            {
                p_renderer->extra_ver = top;
            }
        }
    }
    left   *= scale_x;
    width  *= scale_x;
    top    *= scale_y;
    height *= scale_y;
    
    x_min_new = position->x + (left * cos_angle);
    y_min_new = position->y - (left * sin_angle);
    
    p1.x = x_min_new + (top * sin_angle);
    p1.y = y_min_new + (top * cos_angle);
    
    p2.x = p1.x + (width * cos_angle);
    p2.y = p1.y - (width * sin_angle);
    
    p3.x = p2.x - (height * sin_angle);
    p3.y = p2.y - (height * cos_angle);

    p4.x = p1.x - (height * sin_angle);
    p4.y = p1.y - (height * cos_angle);

    x_max_new = max4(p1.x, p2.x, p3.x, p4.x);
    x_min_new = min4(p1.x, p2.x, p3.x, p4.x);
    y_max_new = max4(p1.y, p2.y, p3.y, p4.y);
    y_min_new = min4(p1.y, p2.y, p3.y, p4.y);

    if(x_min > x_min_new)
    {
        x_min = x_min_new;
        textBBox->left = x_min;
    }
    if(x_max < x_max_new)
    {
        x_max = x_max_new;
    }
    if(y_min > y_min_new)
    {
    	y_min = y_min_new;
    }
    if(y_max < y_max_new)
    {
    	y_max = y_max_new;
        textBBox->top = y_max;
    }

    /* calc new width and height of bounding box */
    textBBox->height = textBBox->top - y_min;
    textBBox->width  = x_max - textBBox->left;
    /* store new "end position" of bounding box */
    if(SVG_FALSE == p_renderer->is_vertical)
    {
        advance += (p_renderer->distance_x + p_renderer->kerning);
        position->x += advance * cos_angle * p_renderer->scale_x;
        position->y -= advance * sin_angle * p_renderer->scale_x;
    }
    else
    {
        advance += (p_renderer->distance_y + p_renderer->kerning);
        position->x -= advance * sin_angle * p_renderer->scale_y;
        position->y -= advance * cos_angle * p_renderer->scale_y;
    }
}

SVGError grlft_get_glyphmetrics( SVGFontContext* p_ctx,
								SVGUint32	*theglyph )
{
	SVGError err = SVG_NO_ERROR;
	get_char_info_struct char_info = { 0 };
	SVGFontFreeTypeRenderer   	*p_renderer   = (SVGFontFreeTypeRenderer*)
											p_ctx->p_fontRendererContext;

	char_info.index	=	p_renderer->cached_index;
	char_info.flags	=	p_renderer->cached_rendrflags;
	
	err	=	local_get_bitmap(	p_ctx,
								&char_info,
								1,0,NULL,theglyph);
	return err;								
}


SVGError grlft_get_stringunicodelist( void *ctx,
								   get_char_info_struct* char_info,
								   SVGUint32		textlength )
{
	SVGError	ret	= SVG_NO_ERROR;
	SVGUint8	*p_theChar = char_info->theChar;
	SVGUint8	*pchar;	
	SVGUint32	loop = 0;
	SVGFontContext* p_ctx = (SVGFontContext*)ctx;

	SVGFontFreeTypeRenderer   	*p_renderer        = (SVGFontFreeTypeRenderer*)p_ctx->p_fontRendererContext;

	if( NULL != char_info->CharIndexList )
	{
		while(char_info->theChar < (p_theChar + textlength))
		{
		    char_info->theChar   = local_decode_char(char_info);

			if ((0 == char_info->theGlyph) && (SVG_TRUE == p_renderer->useReplacementGlyph))
			{
				if(p_ctx->new_replacement_char == SVG_TRUE)
				{
					pchar = char_info->theChar;
					char_info->theChar = p_ctx->replacement_char;
					char_info->theChar = local_decode_char(char_info);
															
					p_renderer->replacementGlyph = char_info->theGlyph;
					p_ctx->new_replacement_char = SVG_FALSE;																  
					char_info->theChar = pchar;
				}
				char_info->theGlyph = p_renderer->replacementGlyph;
			}

			if( loop < (textlength * 2))
			{
				char_info->CharIndexList[loop] = char_info->decodedChar;
			}

			char_info->theChar = &char_info->theChar[1];
			++loop;
		}
		char_info->textlength = loop;
	}
	else
	{
		ret	=	SVG_POINTER_NULL;
	}

	return ret;
}

/* search the given char from font cache and returns the bitmap of the char */
static SVGError local_get_bitmap( SVGFontContext* p_ctx,
                           get_char_info_struct* char_info,
                           SVGUint32 loop,
                           SVGBoolean cache_bitmap,
                           SVGBoolean *texture_update,
                           SVGUint32	*theglyph)
{
	SVGUint32                   retry = 0;
	SVGError                  	ret                = SVG_NO_ERROR;
	SVGFontFreeTypeRenderer   	*p_renderer        = (SVGFontFreeTypeRenderer*)p_ctx->p_fontRendererContext;
    FT_GlyphSlot              slot                = NULL;
    UTIL_hash_elem_t		   entry_to_find	  = {NULL, 0, NULL};
    UTIL_hash_elem_t		  *found_entry		  = NULL;
    SVGUint8*                  p_theChar          = NULL;
    grl_internal_cache_struct *cache_current      = NULL;
    grl_internal_cache_struct  cache_entry        = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
#ifdef HAS_OPENVG
               ,0
#endif
               ,0,0,0,0,0,0,0,0
                                                    };
    FT_BitmapGlyph             bitmap             = {0};
    FT_Bitmap*                 source             = NULL;
	char_info->flags							 |= FT_LOAD_NO_BITMAP;//necessary for .ttc fonts

	if(p_renderer->enableshapingeng == SVG_TRUE)
	{
		if( theglyph != NULL )
		{
			char_info->theGlyph = *theglyph;
		}
		else
		{		
			char_info->theGlyph = p_renderer->p_grl_shpeng_fp_table->GetGlyphCharIndex(
												(void*)p_renderer->shared->p_shape_eng,
												loop);	
		}
	}
	else
	{
	    char_info->theChar   = local_decode_char(char_info);
	    char_info->theGlyph  = FT_Get_Char_Index( p_renderer->shared->face,
	                                              char_info->decodedChar );
	}

    /* when the decoded char is not valid and a replacement char is defined 
       replace the invalid glyph by the replacement char */
    if ((0 == char_info->theGlyph) && (SVG_TRUE == p_renderer->useReplacementGlyph))
    {
        if(p_ctx->new_replacement_char == SVG_TRUE)
        {
            p_theChar = char_info->theChar;
            char_info->theChar = p_ctx->replacement_char;
            char_info->theChar = local_decode_char(char_info);
                                                    
            char_info->theGlyph  = FT_Get_Char_Index( p_renderer->shared->face,
                                                      char_info->decodedChar );
            p_renderer->replacementGlyph = char_info->theGlyph;
            p_ctx->new_replacement_char = SVG_FALSE;                                                                  
            char_info->theChar = p_theChar;
        }
        char_info->theGlyph = p_renderer->replacementGlyph;
   }

    /* check if glyph is valid */
    if (char_info->theGlyph != 0)
    {

    	p_renderer->shared->cache_current = NULL;
    	/*create hash element*/
    	cache_entry.glyph = char_info->theGlyph;
    	cache_entry.font_size = p_renderer->font_size;
    	cache_entry.index = char_info->index;
    	entry_to_find.key = &cache_entry;
    	entry_to_find.key_len = sizeof(SVGUint32)*3;
    	entry_to_find.value = &cache_entry;

        /* find the glyph in the hash table */
    	found_entry = UTIL_sea_hsh(p_renderer->shared->hash_tbl_glyphs,&entry_to_find);

		if( found_entry != NULL)
		{/*cash entry found*/
			p_renderer->shared->cache_current = found_entry->value;
			cache_current = p_renderer->shared->cache_current;
			cache_current->is_in_use = SVG_TRUE;

			if ((cache_current->cached_bitmap == SVG_FALSE) && (cache_bitmap == SVG_TRUE) && (0 != cache_current->width))
			{
				/*need to cache glyph*/
				bitmap = (FT_BitmapGlyph)cache_current->ft_glyph;
				ret = local_add_wallpaper_glyph(p_renderer, cache_current, bitmap->bitmap.buffer);
				if(ret == SVG_NO_ERROR)
				{
					cache_current->cached_bitmap = SVG_TRUE;
					*texture_update = SVG_TRUE;
					if (cache_current->glyph_done != SVG_TRUE)
					{
						FT_Done_Glyph(cache_current->ft_glyph);
						cache_current->glyph_done = SVG_TRUE;
					}
				}
				else
				{
					/* prevent drawing, because there is nothing to draw */
					cache_current = NULL;
					grlftSetError(p_ctx, ret);
				}
			}
			/* move cache entry to first position only when bitmap is cached successfully or
			 * when bitmap has not to be cached(bbox/maxChar)
			 */
			if ((SVG_NO_ERROR == ret) && (cache_current != NULL))
			{
				if(cache_current->prev != NULL)
				{/*move entry to the first position*/
					cache_current->prev->next = cache_current->next;
					if(cache_current->next != NULL)
					{/*if no last entry*/
						cache_current->next->prev = cache_current->prev;
					}
					cache_current->prev = NULL;
					cache_current->next = *p_renderer->shared->cache;
					(*p_renderer->shared->cache)->prev = cache_current;
					*p_renderer->shared->cache = cache_current;
				}
			}
		}
		else
        {/* cache entry not found */
            /* create new cache entry */
			cache_current = (grl_internal_cache_struct*)GRL_malloc_1D_resource(sizeof(grl_internal_cache_struct));
            found_entry                       = (UTIL_hash_elem_t*)GRL_malloc_1D_resource(sizeof(UTIL_hash_elem_t));

			if( (NULL != cache_current) && (NULL != found_entry) )
			{
				memset(cache_current, 0, sizeof(grl_internal_cache_struct));

	            if(0 == FT_Load_Glyph( p_renderer->shared->face,
	                                   char_info->theGlyph,
	                                   (SVGInt32)char_info->flags ))
	            {	
					cache_current->glyph = char_info->theGlyph;
	                FT_Get_Glyph( p_renderer->shared->face->glyph, &cache_current->ft_glyph );
					
	                if(0 == char_info->index)
	                {
	                    FT_Glyph_Stroke( &cache_current->ft_glyph, p_renderer->shared->stroker, TRUE );
	                }
	                    
	                if (0 == FT_Glyph_To_Bitmap(&cache_current->ft_glyph, 
	                                           FT_RENDER_MODE_NORMAL,
	                                           0, 
	                                           1))
	                {            
	                	bitmap = (FT_BitmapGlyph)cache_current->ft_glyph;
	                    source = &bitmap->bitmap;
	                    slot = p_renderer->shared->face->glyph;
	
	                    if (NULL != slot)
	                    {
	                    cache_current->rsb_delta   = (SVGInt32)slot->rsb_delta;
	                    cache_current->lsb_delta   = (SVGInt32)slot->lsb_delta;
	                    cache_current->top_hor     = (SVGInt32)bitmap->top;
	                    cache_current->left_hor    = (SVGInt32)bitmap->left;
	                    cache_current->width       = (SVGUint32)source->width;
	                    cache_current->height      = (SVGUint32)source->rows;
	                    cache_current->advance_hor = (SVGInt32)((slot->metrics.horiAdvance + 32) >> 6);
	                    cache_current->advance_ver = (SVGInt32)((slot->metrics.vertAdvance + 32) >> 6);

	                    if(p_renderer->enableshapingeng == SVG_TRUE)
	                    {
		                    cache_current->horz_bearingx	=	slot->metrics.horiBearingX;
		                    cache_current->horz_bearingy	=	slot->metrics.horiBearingY;
		                    cache_current->vert_bearingx	=	slot->metrics.vertBearingX;
		                    cache_current->vert_bearingy	=	slot->metrics.vertBearingY;
		                    cache_current->metrics_width	=	slot->metrics.width;
		                    cache_current->metrics_height	=	slot->metrics.height;
							cache_current->metrics_advance_x	=	slot->advance.x;
							cache_current->metrics_advance_y	=	slot->advance.y;
	                    }
	                    
	                    cache_current->top_ver     = ((SVGInt32)cache_current->advance_ver - (SVGInt32)cache_current->height) >> 1;
	                    cache_current->left_ver    = -((SVGInt32)cache_current->width >> 1);
	                    cache_current->font_size   = p_renderer->font_size;
	                    cache_current->index       = char_info->index;

							/*create hash table entry*/
							found_entry->key 		= cache_current;
							found_entry->key_len	= sizeof(SVGUint32)*3 ;
							found_entry->value 		= cache_current;
							if ((0 != cache_current->width) && (cache_bitmap == SVG_TRUE))
							{
								ret = local_add_wallpaper_glyph(p_renderer, cache_current, source->buffer);
								if(ret == SVG_NO_ERROR)
								{
									cache_current->cached_bitmap = SVG_TRUE;
									*texture_update = SVG_TRUE;
								}
								else
								{
									grlftSetError(p_ctx, ret);
								}
							}
							/* cache only when no error has occurred -> glyph could be added to wallpaper
							 * or should not be added to wallpaper
							 */
							if( SVG_NO_ERROR == ret)
							{
								while( (NULL == UTIL_add_hsh( p_renderer->shared->hash_tbl_glyphs , found_entry )) && (retry < max_retries))
								{
									retry++;
									if(TOO_MANY_GLYPHS_FOR_CACHE == local_cleanup_cache(p_renderer))
									{
										retry = max_retries;
										ret = TOO_MANY_GLYPHS_FOR_CACHE;
									}
								}
							}
	                    }
	                    else
	                    {
	                    	ret = SVG_POINTER_NULL;
	                    }

						if(SVG_NO_ERROR == ret)
						{
							/* insert element into cache chain only when everything was fine */
							p_renderer->shared->cache_current = cache_current;
							cache_current->next = *p_renderer->shared->cache;
							if( *p_renderer->shared->cache != NULL)
							{
								(*p_renderer->shared->cache)->prev = cache_current ;
							}
							*p_renderer->shared->cache = cache_current;
							(*p_renderer->shared->cache_count)++;
							cache_current->is_in_use = SVG_TRUE;
							/*delete the glyph if: already cached, a white space, or drawing can never happen*/
							/* free glyph only when the bitmap should be cached AND has really been cached */
							if (cache_bitmap == SVG_TRUE)
							{
								FT_Done_Glyph(cache_current->ft_glyph);
								cache_current->glyph_done = SVG_TRUE;
							}
						}
						else
						{
							/* destroy cache element, because something has went wrong
							 * and it can not be used
							 */
							FT_Done_Glyph(cache_current->ft_glyph);
							GRL_free_1D_resource(cache_current);
							cache_current = NULL;
						}
					}
					else
					{
						FT_Done_Glyph(cache_current->ft_glyph);
						GRL_free_1D_resource(cache_current);
						cache_current = NULL;
	               	}	               
	            }
                GRL_free_1D_resource(found_entry);
	        }
	        else
	        {
				if ( NULL != cache_current )
	            {
					GRL_free_1D_resource( cache_current );
					cache_current = NULL;
	            }
	            if ( NULL != found_entry )
                {
                    GRL_free_1D_resource( found_entry );
                }
	        	ret = SVG_OUT_OF_MEMORY;
			    SVG_FNT_F("SVG_OUT_OF_MEMORY IN GRL_GET_BITMAP");
	        }
    	}

		if( theglyph == NULL )
		{
	        /* calculate kerning offset */
	        local_calc_kerning_offset(p_ctx, char_info);
	        
	        if(NULL != cache_current)
	        {
	            if(SVG_TRUE != p_renderer->is_vertical)
	            {
	                cache_current->advance = cache_current->advance_hor;
	                cache_current->left    = cache_current->left_hor;
	                if(p_renderer->enableshapingeng == SVG_TRUE)
	                {

	                	SVGInt32 glyph_Y_offset;

						p_renderer->p_grl_shpeng_fp_table->GetGlyphOffsets(
												(void*)p_renderer->shared->p_shape_eng,
													loop,
													NULL,
													&glyph_Y_offset);
					        			 
						cache_current->top     = cache_current->top_hor  + glyph_Y_offset;
		            }
		            else
		            {
						cache_current->top     = cache_current->top_hor;
		            }

	                if (cache_current->top > p_renderer->top_most)
	                {
	                    p_renderer->top_most = cache_current->top;
	                }
	                if ((cache_current->height - cache_current->top) > p_renderer->bottom_most)
	                {
	                    p_renderer->bottom_most = cache_current->height - cache_current->top;
	                }
	                p_renderer->offset_y = (cache_current->top - cache_current->height) + p_renderer->advance_y;
	                p_renderer->advance_y = -(cache_current->top - cache_current->height);
	    
	                if(loop == 0) 
	                {
	                    if ((p_ctx->settings & SVG_FONT_NO_ENCLOSING_SPACES) != 0)
	                    {
	                        p_renderer->offset_x = 0.0f;
	                    }
	                    else
	                    {
	                        p_renderer->offset_x = cache_current->left + p_renderer->kerning + p_renderer->advance_x;
	                    }
	                }
	                else
	                {
	                    p_renderer->offset_x = cache_current->left + p_renderer->kerning + p_renderer->advance_x + p_renderer->distance_x;
	                }
	                p_renderer->advance_x = cache_current->advance - cache_current->left;
	    
	            }
	            else
	            {
	                cache_current->advance = cache_current->advance_ver;
	                cache_current->top     = cache_current->top_ver;
	                cache_current->left    = cache_current->left_ver;
	                if (cache_current->left < p_renderer->left_most)
	                {
	                    p_renderer->left_most = cache_current->left;
	                }
	                if (fabs(cache_current->left + cache_current->width) > p_renderer->right_most)
	                {
	                    p_renderer->right_most = (SVGFloat)fabs(cache_current->left + cache_current->width);
	                }
	     
	                p_renderer->offset_x = cache_current->left - p_renderer->advance_x ;
	                p_renderer->advance_x = cache_current->left;
	    
	                if(loop == 0)
	                {
	                    if ((p_ctx->settings & SVG_FONT_NO_ENCLOSING_SPACES) != 0)
	                    {
	                        p_renderer->offset_y = -cache_current->height;
	                    }
	                    else
	                    {
	                        p_renderer->offset_y = -(cache_current->top + cache_current->height + p_renderer->advance_y);
	                    }
	                }
	                else
	                {
	                    p_renderer->offset_y = -(cache_current->top + cache_current->height + p_renderer->advance_y + p_renderer->distance_y);
	                }
	              
	                p_renderer->advance_y = cache_current->advance - (cache_current->top + cache_current->height);
	            }
	            char_info->rsb_delta = cache_current->rsb_delta;
	            char_info->lsb_delta = cache_current->lsb_delta;
	        }
	        else
	        {
			    SVG_FNT_E("CACHE_ELEMENT_NOT_AVAILABLE IN GRL_GET_BITMAP");
	        }
	    }
    }
    else
    {
        ret = SVG_NULL_CHARACTER;
    }
    return ret;
}


/**
 * decode the character code in dependency of the running length
 * which is given by the encoding
 
 */
static SVGUint8* local_decode_char( get_char_info_struct* char_info)
{
    SVGUint8*   p_theChar       = char_info->theChar;
    SVGEncoding charMap         = char_info->encoding;
    SVGUint8    uni8_bytes      = 0;
    SVGUint8    uni8_first      = 0;
    SVGUint8    uni8_mask       = 0;
    SVGUint8    loop            = 0;

    switch (charMap)
    {
        case SVG_ENCODING_ASCII      :
            char_info->decodedChar = *p_theChar;
            break;
        case SVG_ENCODING_SJIS       :
            /* 1 or 2 bytes per char */
            p_theChar++;
            if ( ((*p_theChar & 0xf0) == 0x80) ||
                 ((*p_theChar & 0xf0) == 0x90) ||
                 ((*p_theChar & 0xf0) == 0xE0) ||
                 ((*p_theChar & 0xf0) == 0xF0))
            {
                /* two byte coding */
                char_info->decodedChar = ((SVGUint32)*p_theChar) << 8;
                p_theChar--;
                char_info->decodedChar += *p_theChar;
                p_theChar++;
            }
            else
            {
                /* only one byte */
                p_theChar--;
                char_info->decodedChar = *p_theChar;
            }
            break;
        case SVG_ENCODING_UTF8       :
            /* java specific */
            if ((*p_theChar == 0xC0) && (p_theChar[1] == 0x80))
            {
                char_info->decodedChar = 0x00;
                p_theChar++;
            }
            else
            {
                /* skip middle information */
                while ((*p_theChar & 0xC0) == 0x80)
                {
                    p_theChar++;
                }

                /* get count of following bytes and set mask */
                uni8_first = *p_theChar;
                uni8_mask  = 0xFF;
                uni8_bytes = 0;
                while ((uni8_first & 0x80) != 0)
                {
                    uni8_first <<= 1;
                    uni8_mask  >>= 1;
                    uni8_bytes++;
                }

                if (uni8_bytes == 0)
                {
                    /* ASCII value */
                    char_info->decodedChar = *p_theChar;
                }
                else
                {
                    /* UTF-8 code (support 6 bytes) */
                    for (loop = 0; loop < uni8_bytes; loop++)
                    {
                        if (loop == 0)
                        {
                            char_info->decodedChar = (SVGUint32)(*p_theChar & uni8_mask);
                        }
                        else
                        {
                            char_info->decodedChar += (SVGUint32)(*p_theChar & 0x3F);
                        }

                        if (loop < (uni8_bytes - 1))
                        {
                            char_info->decodedChar <<= 6;
                        }
                        p_theChar++;
                    }
                    p_theChar--;
                }
            }
            break;

        default                      :
            /* default is unicode: 2 bytes per char */
            char_info->decodedChar = *p_theChar;
            p_theChar++;
            char_info->decodedChar += ((SVGUint32)*p_theChar) << 8;
            break;
    }

    /* end of encoding */
    return p_theChar;
}


static void local_setFontMask(const SVGFontContext* ctx, get_char_info_struct *char_info)
{
    SVGFontFreeTypeRenderer   *p_renderer         = (SVGFontFreeTypeRenderer*)
                                                    ctx->p_fontRendererContext;
#ifdef HAS_OPENVG
	VGImage vgWallpaper = VG_INVALID_HANDLE;
#endif
    if( 0 == p_renderer->texture)
    {
        if(((ctx->settings & SVG_FONT_NO_ENCLOSING_SPACES) == 0) ||
           (SVG_ORIGIN_BASELINE == ctx->base_line))
        {
            /* do rendering only when origin is Baseline or 
               no_enclosing_spaces is not enabled, because then
               ascender and descender can be used. in any other combination
               the topmost glyph position has to be determined by iterating 
               over the whole string. So drawing is done at the end of the outer loop */
#ifdef HAS_OPENVG
            if(SVG_USE_OPENVG == p_renderer->drawingAPI)
            {
                local_render_vg_glyph(ctx);
            }
#endif
#ifdef HAS_OPENGLES
            if(SVG_USE_OPENGLES_1 == p_renderer->drawingAPI)
            {
                local_render_gl_glyph(ctx);
            }                         
#endif
        }
    }
    else
    {
        /* here starts copying to fontBitmap texture */
#ifdef HAS_OPENVG
        if(SVG_USE_OPENVG == p_renderer->drawingAPI)
        {
			if(char_info->index ==0 )
				vgWallpaper = p_renderer->shared->vgWallpaperStrk;
			else
				vgWallpaper = p_renderer->shared->vgWallpaperFill;
			if(0 != (p_renderer->shared->cache_current->width * p_renderer->shared->cache_current->height))
			{    
				 vgCopyImage((VGImage)p_renderer->texture,
							 (SVGInt32)(char_info->next_pos.x + (p_renderer->shared->cache_current->left * p_renderer->scale_x)),
							 (SVGInt32)(char_info->next_pos.y + 
										((p_renderer->shared->cache_current->top - p_renderer->shared->cache_current->height) * p_renderer->scale_y) +
										p_renderer->origin_hor),
							 vgWallpaper,
							 (SVGInt32)p_renderer->shared->cache_current->source_x,
							 (SVGInt32)p_renderer->shared->cache_current->source_y,
							 (SVGInt32)p_renderer->shared->cache_current->width,
							 (SVGInt32)p_renderer->shared->cache_current->height,
							 VG_FALSE);
			}
        }
#endif
#ifdef HAS_OPENGLES
        if(SVG_USE_OPENGLES_1 == p_renderer->drawingAPI)
        {     
			glBindTexture(GL_TEXTURE_2D,p_renderer->texture);
        	local_store_gl_bitmap(ctx, char_info, p_renderer->shared->cache_current->index);            	
        }
#endif
    }
}


static void local_calc_kerning_offset( SVGFontContext* ctx,
                                get_char_info_struct* char_info)
{
    FT_Vector               vec         = {0, 0};
    SVGFontFreeTypeRenderer *p_renderer = (SVGFontFreeTypeRenderer*)
                                          ctx->p_fontRendererContext;

    p_renderer->kerning = 0.0f;
    if(SVG_FALSE == p_renderer->is_first_char)
    {
        if (((ctx->settings & SVG_FONT_KERNING) != 0) &&
            (char_info->theGlyph != 0) && (char_info->prevGlyph != 0))
        {
            if (FT_Get_Kerning( p_renderer->shared->face,
                                char_info->prevGlyph,
                                char_info->theGlyph,
                                p_renderer->kern_mode,
                                &vec ) == 0)
            {
                p_renderer->kerning = ((SVGFloat)(vec.x))  / 64.0f;
            }
        }
        /* add correction factor for hinting also when kerning is not activated */
        if ( char_info->prev_rsb_delta - char_info->lsb_delta >= 32.0f )           
        {
            p_renderer->kerning -= 1.0f;
        }
    }

    char_info->prevGlyph      = char_info->theGlyph;
    char_info->prev_rsb_delta = char_info->rsb_delta;
    p_renderer->is_first_char = SVG_FALSE;
}

static SVGUint32 local_drawFontWithCursorInfo(SVGFontContext* ctx,
                                       SVGPoint* position,
                                       SVGFloat* angle,
                                       SVGUint8* text,
                                       SVGUint32 textLength,
                                       SVGPoint* cursorPosArray,
                                       SVGBoolean draw_ext_glyphs,
                                       SVGBBox  *textBBox)
{
    SVGUint32               error_api          = 0;
    SVGPoint*               cursorPos          = cursorPosArray ;
    SVGUint32               loop               = 0;
    SVGUint32               hbGlyphCount	   = 0;
    get_char_info_struct    char_info          = {NULL, 0, 0, 0, 0, 0, {0,0}, SVG_ENCODING_MIN, 0, 0, 0, 0, NULL, 0};
    SVGError                error              = SVG_NO_ERROR;
    SVGBoolean              term_found         = SVG_FALSE;
    SVGUint32               stroke_width       = 0;
    SVGUint32               max_chars          = 0;
    SVGUint32               max_width          = 0xffffffff;
    SVGUint32               max_height         = 0xffffffff;
    SVGBBox                 localTextBBox      = {0.0f, 0.0f, 0.0f, 0.0f};
    SVGFloat                cos_angle          = SVG_COSINE_LUT[(SVGUint32)*angle];
    SVGFloat                sin_angle          = SVG_SINE_LUT[(SVGUint32)*angle];
    SVGPoint                cursor_position    = {0.0f, 0.0f};
    /* set current char is first char for calc_maximum_extension 
       for considering left and right space when NO_ENCLOSING_SPACES is not set */
    SVGBoolean              first_or_last_char = SVG_TRUE;
    SVGFontFreeTypeRenderer *p_renderer        = (SVGFontFreeTypeRenderer*)
                                                 ctx->p_fontRendererContext;
    SVGFloat                distance           = ctx->letter_distance;                                                  
#ifdef HAS_OPENGLES1
    SVGFloat                stroke_width_f     = 0.0f;
    SVGFloat                mv_matrix[16]         = {0.0f};
#endif    
    /* set it to false, when origin is BASELINE	and OUTLINE or 2COLOR are not set */
    SVGBoolean              draw_text_final    = SVG_TRUE;
    SVGBoolean              draw_text    	   = SVG_FALSE;
    SVGUint32               l_index              = 0;
#ifdef HAS_OPENVG
    SVGFloat                matrix_tmp[9]      = {0.0f};
    SVGFloat                vg_master_matrix[9]= {0.0f};
#endif    
    SVGBoolean              stored             = SVG_FALSE;
    grl_late_draw_struct    *late_glyph        = NULL;
    SVGUint32                glyphs_to_draw    = 0;
    SVGFloat                ext_angle          = 0.0f;
#ifdef HAS_OPENGLES2
    SVGInt32	            current_program    = 0;
#endif
	SVGBoolean              is_update_fill     = SVG_FALSE; /*indicates texture update of fill glyphs*/
    SVGBoolean              is_update_strk     = SVG_FALSE; /*indicates texture update of strk glyphs*/

#if defined (HAS_OPENGLES1) || defined (HAS_OPENVG)
    SVGFloat                current_color[4]   = {0.0f};                                                
    SVGFloat                matrix[16]         = {0.0f};
#endif    
	SVGUint32				strk_baseline_gap  = 0;
	SVGBoolean				cache_bitmap 	   = SVG_FALSE;
	SVGBoolean				add_new_glyph	   = SVG_FALSE;
    /* ensure that no drawing api error is there before */
#ifdef HAS_OPENGLES
	GLfloat  				*texCoords			= NULL ;
	GLfloat  				*vertx     			= NULL;
	GLint	 				*first       		= NULL;
	GLsizei	 				*glyphEdges  		= NULL;
	
	GLfloat  				prev_off_x 			= 0;
	GLfloat   				prev_off_y 			= 0;
	SVGBoolean 				drawMultSupported   = SVG_FALSE;
    /* ensure that no drawing api error is there before */
    if(SVG_USE_OPENGLES_1 == p_renderer->drawingAPI)
    {      
        glGetError();
    }
#endif        
#ifdef HAS_OPENVG
    if(SVG_USE_OPENVG == p_renderer->drawingAPI)
    {      
        vgGetError();
    }
#endif        

    SVG_FNT_D("Getting glyphs for string %s with text length of %d and size of %d"
    		, text, textLength,  p_renderer->font_size);
	
	if( NULL == textBBox )
	{
		cache_bitmap = SVG_TRUE;
	}
	if ( draw_ext_glyphs == SVG_TRUE)
	{
		draw_text_final = SVG_FALSE;
	}

    p_renderer->distance_x = 0.0f;
    p_renderer->distance_y = 0.0f;
    p_renderer->scale_x    = 1.0f;
    p_renderer->scale_y    = 1.0f;
    p_renderer->extra_ver  = 0.0f;
    if ((ctx->settings & SVG_FONT_SCALE) == SVG_FONT_SCALE)
    {
        p_renderer->scale_x = ctx->scale_x;
        p_renderer->scale_y = ctx->scale_y;
    }
    p_renderer->sin_angle = sin_angle;
    p_renderer->cos_angle = cos_angle;
    if ( textLength > 0)
    {
        /* either for getBBox or for getMaxChars */
        if(NULL != textBBox)
        {
            /* get values for max_char calculation */
            if((0 != textBBox->width) || (0 != textBBox->height))
            {
                max_width  = (SVGUint32)(textBBox->width + 0.5f);
                max_height = (SVGUint32)(textBBox->height  + 0.5f);
                textBBox->width  = 0;
                textBBox->height = 0;
            }
/* do not set the fontmask when boundingBox or getMaxChars is called */
			draw_text_final = SVG_FALSE;
        }
        /* special case for enabling localSetFontMask for fontBitmap */
        if( 0 != p_renderer->texture)
        {
            draw_text = SVG_TRUE;
			draw_text_final = SVG_FALSE;
        }
        char_info.encoding  = ctx->encoding;
        p_renderer->additional_space = 0.0;
        p_renderer->origin_hor = 0.0;
        p_renderer->origin_ver = 0.0f;
        p_renderer->top_most = 0;
        p_renderer->bottom_most = 0;

        /* lock shared context now due to changes in freetype settings */

        p_renderer->start = 1;
        p_renderer->end   = 2;
        /* prepare the stroker with the linewidth when 2-color or outline is enabled */
 
	   if (((ctx->settings & SVG_FONT_2COLOR) != 0) && (SVG_TRUE == p_renderer->stroker_availabe))
        {
            p_renderer->start  = 0;
			strk_baseline_gap = 1;
            if(SVG_TRUE == ctx->WidthSet )
            {
                stroke_width = ctx->ContourWidth;
            }
            else
            {
#ifdef HAS_OPENVG
                if(SVG_USE_OPENVG == p_renderer->drawingAPI)
                {
                    stroke_width = (SVGUint32)vgGeti( VG_STROKE_LINE_WIDTH );
                }
#endif            
                if(SVG_USE_OPENGLES_1 == p_renderer->drawingAPI)
                {
#ifdef HAS_OPENGLES1
                    glGetFloatv(GL_LINE_WIDTH, &stroke_width_f);
                    stroke_width = (SVGUint32)stroke_width_f;
#else
                    stroke_width = ctx->ContourWidth;
#endif            
                }
            }
            if(stroke_width < 2)
            {
                stroke_width = 2;
            }

            FT_Stroker_Set(p_renderer->shared->stroker,
                           (SVGInt64)stroke_width << 5,
                           FT_STROKER_LINECAP_ROUND,
                           FT_STROKER_LINEJOIN_MITER,
                           (SVGInt64) 3 << 15 );
            p_renderer->additional_space = (SVGFloat)stroke_width;

                if ((ctx->settings & SVG_FONT_OUTLINE_SPACE_EXTENSION) != 0)
            {
                ctx->letter_distance += p_renderer->additional_space;
            }
        }
        else
        {
            if (((ctx->settings & SVG_FONT_OUTLINE) != 0) && (SVG_TRUE == p_renderer->stroker_availabe))
            {
                p_renderer->start = 0;
                p_renderer->end   = 1;
				strk_baseline_gap = 1;
                if(SVG_TRUE == ctx->WidthSet )
                {
                    stroke_width = ctx->ContourWidth;
                }
                else
                {
#ifdef HAS_OPENVG
                    if(SVG_USE_OPENVG == p_renderer->drawingAPI)
                    {
                        stroke_width = (SVGUint32)vgGeti( VG_STROKE_LINE_WIDTH );
                    }
#endif                
                    if(SVG_USE_OPENGLES_1 == p_renderer->drawingAPI)
                    {
#ifdef HAS_OPENGLES1
                        glGetFloatv(GL_LINE_WIDTH, &stroke_width_f);
                        stroke_width = (SVGUint32)stroke_width_f;
#else
                        stroke_width = ctx->ContourWidth;
#endif            
                    }
                }

                if(stroke_width < 2)
                {
                    stroke_width = 2;
                }
                FT_Stroker_Set(p_renderer->shared->stroker,
                               (SVGInt64)stroke_width << 5,
                               FT_STROKER_LINECAP_ROUND,
                               FT_STROKER_LINEJOIN_MITER,
                               (SVGInt64)3<< 15 );
                p_renderer->additional_space = (SVGFloat)stroke_width;
                if ((ctx->settings & SVG_FONT_OUTLINE_SPACE_EXTENSION) != 0)
                {
                    ctx->letter_distance += p_renderer->additional_space;
                }
            }
        }
		
		if(SVG_ORIGIN_BASELINE == ctx->base_line)
        {
			p_renderer->origin_hor = (SVGFloat)((stroke_width >> 1) + strk_baseline_gap);
        }
        else if(SVG_ORIGIN_BOTTOM == ctx->base_line)
        {
            p_renderer->origin_hor = p_renderer->descender + (SVGFloat)(stroke_width >> 1) +1;
        }
		else if( SVG_ORIGIN_TOP == ctx->base_line )
		{
            p_renderer->origin_hor = -(p_renderer->ascender + (SVGFloat)(stroke_width >> 1) -1);
		}
		
		if( (p_renderer->stroke_width != stroke_width) &&
			(p_renderer->stroke_width != 0) &&
			(stroke_width != 0))
		{
			local_del_strk_glyphs( ctx );
		}
        p_renderer->stroke_width = stroke_width;
        
		char_info.flags  = FT_LOAD_DEFAULT;
        /* is hinting */
        if ((ctx->settings & SVG_FONT_HINTING) == SVG_FONT_HINTING)
        {
            p_renderer->kern_mode = FT_KERNING_DEFAULT;
            char_info.flags |= FT_LOAD_FORCE_AUTOHINT;
        }
        else
        {
            p_renderer->kern_mode = FT_KERNING_UNFITTED;
            char_info.flags |= FT_LOAD_NO_HINTING;
        }

        p_renderer->is_vertical = SVG_FALSE;
        p_renderer->distance_x = ctx->letter_distance;
        if ((ctx->settings & SVG_FONT_VERTICAL) == SVG_FONT_VERTICAL)
        {
            p_renderer->distance_y = ctx->letter_distance;
            p_renderer->distance_x = 0.0f;
            p_renderer->is_vertical = SVG_TRUE;
            if(SVG_ORIGIN_TOP == ctx->base_line)
            {
                p_renderer->origin_ver = 0.5f * (SVGFloat)(p_renderer->left + p_renderer->right);
            }
            else
            {
                if(SVG_ORIGIN_BOTTOM == ctx->base_line)
                {
                    p_renderer->origin_ver = -0.5f * (SVGFloat)(p_renderer->left + p_renderer->right);
                }
            }
            p_renderer->origin_hor = 0.0f;
        }

        if ((ctx->settings & SVG_FONT_OUTLINE_SPACE_EXTENSION) == 0)
        {
            p_renderer->additional_space = 0.0;
        }

        p_renderer->index     = 0;
        cursor_position.x = position[0].x;
        cursor_position.y = position[0].y;

        p_renderer->stroke_color[0] = (SVGFloat)((ctx->ContourColor >> 24) & 0xff) / 255.0f;
        p_renderer->stroke_color[1] = (SVGFloat)((ctx->ContourColor >> 16) & 0xff) / 255.0f;
        p_renderer->stroke_color[2] = (SVGFloat)((ctx->ContourColor >>  8) & 0xff) / 255.0f;
        p_renderer->stroke_color[3] = 1.0f;
        
        p_renderer->fill_color[0] = (SVGFloat)((ctx->BodyColor >> 24) & 0xff) / 255.0f;
        p_renderer->fill_color[1] = (SVGFloat)((ctx->BodyColor >> 16) & 0xff) / 255.0f;
        p_renderer->fill_color[2] = (SVGFloat)((ctx->BodyColor >>  8) & 0xff) / 255.0f;
        p_renderer->fill_color[3] = 1.0f;

#ifdef HAS_OPENGLES
        if ((SVG_USE_OPENGLES_1 == p_renderer->drawingAPI) && (NULL == textBBox))
        {
            eglQuerySurface(eglGetCurrentDisplay(), 
                            eglGetCurrentSurface(EGL_DRAW), 
                            EGL_WIDTH, 
                            &p_renderer->screen_width);
            eglQuerySurface(eglGetCurrentDisplay(), 
                            eglGetCurrentSurface(EGL_DRAW), 
                            EGL_HEIGHT, 
                            &p_renderer->screen_height);
            p_renderer->wallpaperHeight = p_renderer->shared->glWallpaperHeight;
            p_renderer->wallpaperWidth  = p_renderer->shared->glWallpaperWidth;
            glGetIntegerv( GL_UNPACK_ALIGNMENT, &(p_renderer->glUnpackAlignment) );

            glEnable(GL_BLEND);
            glBlendFuncSeparate(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA,GL_ONE, GL_ONE_MINUS_SRC_ALPHA);

            projection[0][0] = 2.0f / p_renderer->screen_width;
            projection[1][1] = 2.0f / p_renderer->screen_height;
#ifdef HAS_OPENGLES1
            glColorMask(GL_TRUE, GL_TRUE, GL_TRUE, GL_FALSE);

          	glLoadMatrixf( &projection[0][0]);

			glEnableClientState(GL_TEXTURE_COORD_ARRAY);
           	glEnableClientState(GL_VERTEX_ARRAY);
            glClientActiveTexture(GL_TEXTURE0);
          	glEnable(GL_TEXTURE_2D);
            glMatrixMode(GL_MODELVIEW);

            glGetFloatv(GL_CURRENT_COLOR, &current_color[0]);
            if(SVG_FALSE == ctx->ColorSet)
            {
                p_renderer->fill_color[0] = current_color[0];
                p_renderer->fill_color[1] = current_color[1];
                p_renderer->fill_color[2] = current_color[2];
            }

            p_renderer->gl_mode = GL_REPLACE;
            if ((ctx->settings & SVG_FONT_OPACITY) == SVG_FONT_OPACITY)
            {
                p_renderer->fill_color[3]   = ctx->opacity;
                p_renderer->stroke_color[3] = ctx->opacity;
                p_renderer->gl_mode = GL_BLEND;
            }
          	glTexEnvx(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, p_renderer->gl_mode);

#else
            glGetIntegerv(GL_CURRENT_PROGRAM, &current_program);
            /* do we use our internal program or an external*/
            if ((ctx->settings & SVG_FONT_USE_CURRENT_PROGRAM) == 0)
            { 
                glUseProgram(p_renderer->shared->programHandle);
                glEnableVertexAttribArray (p_renderer->shared->vertexlocation);
                glEnableVertexAttribArray (p_renderer->shared->texturelocation);
                if ((ctx->settings & SVG_FONT_OPACITY) == SVG_FONT_OPACITY)
			    {
					glUniform1f(p_renderer->shared->opacitylocation, ctx->opacity);
			    }
			    else
			    {
					glUniform1f(p_renderer->shared->opacitylocation, 1.0f);
			    }
			    if ((ctx->settings & SVG_FONT_ANTIALIAS) == SVG_FONT_ANTIALIAS)
			    {
					glUniform1i(p_renderer->shared->noaalocation, 0);
			    }
			    else
			    {
					glUniform1i(p_renderer->shared->noaalocation, 1);
			    }
	        }
	        else
	        {
	        	/*enable the needed states for the external program*/
	        	glEnableVertexAttribArray ((SVGUint32)glGetAttribLocation((SVGUint32)current_program, "inputtexcoord"));
                glEnableVertexAttribArray ((SVGUint32)glGetAttribLocation((SVGUint32)current_program, "position"));
                if ((ctx->settings & SVG_FONT_OPACITY) == SVG_FONT_OPACITY)
			    {
					glUniform1f(glGetUniformLocation((SVGUint32)current_program, "opacity"), ctx->opacity);
			    }
			    else
			    {
					glUniform1f(glGetUniformLocation((SVGUint32)current_program, "opacity"), 1.0f);
			    }
			    if ((ctx->settings & SVG_FONT_ANTIALIAS) == SVG_FONT_ANTIALIAS)
			    {
					glUniform1i(glGetUniformLocation((SVGUint32)current_program, "noantialias"), 0);
			    }
			    else
			    {
					glUniform1i(glGetUniformLocation((SVGUint32)current_program, "noantialias"), 1);
			    }
	        }

#endif
        }
#endif        
#ifdef HAS_OPENVG
        if ((SVG_USE_OPENVG == p_renderer->drawingAPI) && (NULL == textBBox))
        {
            p_renderer->wallpaperWidth  = p_renderer->shared->vgWallpaperWidth;
            p_renderer->wallpaperHeight = p_renderer->shared->vgWallpaperHeight;
            vgSeti(VG_MATRIX_MODE, VG_MATRIX_IMAGE_USER_TO_SURFACE);
            vgGetMatrix(&vg_master_matrix[0]);
            if ((ctx->settings & SVG_FONT_USE_CURRENT_MATRIX) == 0)
            {
                vgLoadIdentity();
            }
            vgGetMatrix(&matrix[0]);
            p_renderer->VGpaint =  vgGetPaint( VG_FILL_PATH );
            vgGetParameterfv(p_renderer->VGpaint, VG_PAINT_COLOR, 4, current_color);
            if(SVG_FALSE == ctx->ColorSet)
            {
                p_renderer->fill_color[0] = current_color[0];
                p_renderer->fill_color[1] = current_color[1];
                p_renderer->fill_color[2] = current_color[2];
            }

            if ((ctx->settings & SVG_FONT_OPACITY) == SVG_FONT_OPACITY)
            {
                p_renderer->fill_color[3]   = ctx->opacity;
                p_renderer->stroke_color[3] = ctx->opacity;
            }
        }
#endif        
        /* when current buffer is not big enough allocate a new one */
        /* buffer is only growing and will be destroyed finally when */
        /* context is destroyed */
        if((2 * textLength) > p_renderer->shared->late_glyphs_size)
        {
            local_destroy_shared_resources(p_renderer);
        }

        /* create necessary shared resources */
        term_found = local_create_shared_resources(p_renderer, textLength);

        /* in case of allocation error throw away the allocated resources */
        if(SVG_TRUE == term_found)
        {
            local_destroy_shared_resources(p_renderer);
            grlftSetError(ctx, TOO_MANY_GLYPHS_FOR_CACHE);
            SVG_FNT_E("TEXT_TOO_LARGE IN LOCAL_DRAWFONTWITHCURSORINFO ");

            return 0;
        }
        
        char_info.next_pos.x = position[0].x;
        char_info.next_pos.y = position[0].y;

        FT_Set_Pixel_Sizes(p_renderer->shared->face, 0, p_renderer->font_size );

/*
  * Finding scripts for the passed string
  * if the complex scripts is found then enable the shaping engine
  * If not found,the simple string is processed without shaping engine
  */
         if((SVG_TRUE == p_renderer->autolangdetect) && \
              ( SVG_TRUE == local_shpeng_get_autolang_config()))
         {
            SVGLangScript script_tag = SVG_SCRIPT_INVALID;
            SVGRenderDirection rendr_direc_Tag = SVG_RENDER_DIRECTION_INVALID;
            char_info.theChar = text;
            char_info.textlength = textLength;
             if( SVG_TRUE == local_shpeng_find_scripttag( &char_info,
                                                                                 (SVGUint32*)&script_tag,
                                                                                 (SVGUint32*)&rendr_direc_Tag) )
             {
                  if(SVG_NO_ERROR ==  local_shpeng_set_langscript_rendrdirection(p_renderer,
                                                                                        script_tag,
                                                                                        rendr_direc_Tag) )
                {
                    p_renderer->enableshapingeng = SVG_TRUE;
                    p_renderer->script_tag = script_tag;
                }
             }
         }

        for(l_index = p_renderer->start; l_index < p_renderer->end; l_index++)
        {
            char_info.prevGlyph   = 0;
            char_info.theGlyph    = 0;
            char_info.index       = l_index;
            char_info.theChar     = text;
            char_info.angle       = angle[0];
            while(char_info.angle < 0.0)
            {
                char_info.angle += MAX_ANGLE;
            }
            while(char_info.angle >= MAX_ANGLE)
            {
                char_info.angle -= MAX_ANGLE;
            }
            term_found            = SVG_FALSE;
            p_renderer->advance_x = 0.0f;
            p_renderer->advance_y = 0.0f;
            p_renderer->kerning   = 0.0f;
            glyphs_to_draw        = 0;
            p_renderer ->is_first_char = SVG_TRUE;
			
            add_new_glyph         = SVG_FALSE;
			if(0 == l_index)
            {
				p_renderer->shared->cache = &p_renderer->shared->cache_strk;
				p_renderer->shared->cache_count = &p_renderer->shared->cache_count_strk;
				p_renderer->shared->hash_tbl_glyphs = p_renderer->shared->hash_tbl_strk;
			}
			else
			{
				p_renderer->shared->cache = &p_renderer->shared->cache_fill;
				p_renderer->shared->cache_count = &p_renderer->shared->cache_count_fill;
				p_renderer->shared->hash_tbl_glyphs = p_renderer->shared->hash_tbl_fill;
			}
#ifdef HAS_OPENGLES
            if ((SVG_USE_OPENGLES_1 == p_renderer->drawingAPI) && (NULL == textBBox))
            {
				if(0 == l_index)
				{
					glBindTexture(GL_TEXTURE_2D,p_renderer->shared->glWallpaperStrk);
					p_renderer->row = p_renderer->shared->glRowStrk;
				}
				else
				{
					glBindTexture(GL_TEXTURE_2D,p_renderer->shared->glWallpaperFill);
					p_renderer->row = p_renderer->shared->glRowFill;
				}
#ifdef HAS_OPENGLES1
                glPushMatrix();
                if(0 == l_index)
                {				
					glColor4f(p_renderer->stroke_color[0], 
                              p_renderer->stroke_color[1], 
                              p_renderer->stroke_color[2], 
                              p_renderer->stroke_color[3]);
                }
                else
                {
					glColor4f(p_renderer->fill_color[0], 
                              p_renderer->fill_color[1], 
                              p_renderer->fill_color[2], 
                              p_renderer->fill_color[3]);
                }
                if ((ctx->settings & SVG_FONT_USE_CURRENT_MATRIX) == 0)
                {
                    glLoadIdentity();
                }
        
                /* additional move for origin */
                if((ctx->settings & SVG_FONT_NO_ENCLOSING_SPACES) == 0)
                {
                    glRotatef((360.0f - char_info.angle), 0., 0., 1. );
                    glTranslatef( p_renderer->origin_ver, p_renderer->origin_hor, 0 );
                    glRotatef( -(360.0f - char_info.angle), 0., 0., 1. );
                }
    
                /* move to starting position */
                glTranslatef (position[0].x, position[0].y, 0.0f);
                glRotatef((360.0f - char_info.angle), 0., 0., 1. );
                if((p_renderer->scale_x < 1.0) || (p_renderer->scale_y < 1.0))
                {
                    glScalef(p_renderer->scale_x, p_renderer->scale_y, 1.0);
                }
#else
                local_matrixIdentity(p_renderer->shared->modelview);
        		/*the color is not needed to be set for external shader*/
        		if ((ctx->settings & SVG_FONT_USE_CURRENT_PROGRAM) == 0)
        		{
	                if(0 == l_index)
	                {
						glUniform4fv(p_renderer->shared->colorlocation, 1, &p_renderer->stroke_color[0]);
	                }
	                else
	                {
						glUniform4fv(p_renderer->shared->colorlocation, 1, &p_renderer->fill_color[0]);
	                }
	            }
                
				/* additional move for origin */
                if((ctx->settings & SVG_FONT_NO_ENCLOSING_SPACES) == 0)
                {
                    local_matrixRotate(p_renderer->shared->modelview,(360.0f - char_info.angle));
                    local_matrixTranslate(p_renderer->shared->modelview, p_renderer->origin_ver, p_renderer->origin_hor);
                    local_matrixRotate(p_renderer->shared->modelview, char_info.angle);
                }               
    
                /* move to starting position */
                local_matrixTranslate (p_renderer->shared->modelview,position[0].x, position[0].y);
                local_matrixRotate(p_renderer->shared->modelview,(360.0f - char_info.angle));
                if((p_renderer->scale_x < 1.0) || (p_renderer->scale_y < 1.0))
                {
                    local_matrixScale(p_renderer->shared->modelview,p_renderer->scale_x, p_renderer->scale_y);
                }
#endif            
				if ( 0 != p_renderer->texture)
				{
					cache_bitmap = SVG_FALSE;
				}
            }
#endif
#ifdef HAS_OPENVG
            if ((SVG_USE_OPENVG == p_renderer->drawingAPI) && (NULL == textBBox))
            {
                vgLoadMatrix(&matrix[0]);
                    
                if(0 == l_index)
                { 
					p_renderer->row = p_renderer->shared->vgRowStrk;
					vgSetParameterfv(p_renderer->VGpaint, VG_PAINT_COLOR, 4, p_renderer->stroke_color);
                }
                else
                {
					p_renderer->row = p_renderer->shared->vgRowFill;
					vgSetParameterfv(p_renderer->VGpaint, VG_PAINT_COLOR, 4, p_renderer->fill_color);
                }
                /* additional move for origin */
                if((ctx->settings & SVG_FONT_NO_ENCLOSING_SPACES) == 0)
                {
                    vgRotate(360.0f - char_info.angle);
                    vgTranslate( p_renderer->origin_ver, p_renderer->origin_hor);
                    vgRotate( -(360.0f - char_info.angle));
              	}
                
                /* move to starting position */
                vgTranslate (position[0].x, position[0].y);
                                vgRotate(360.0f - char_info.angle);
    
                if((p_renderer->scale_x < 1.0) || (p_renderer->scale_y < 1.0))
                {
                    vgScale(p_renderer->scale_x, p_renderer->scale_y);
                }
    
            }
#endif            
	        if((ctx->settings & SVG_FONT_NO_ENCLOSING_SPACES) == 0)
            {
                 /* additional move for origin already handled in drawing API*/                
            }
            else
            {
                cursorPos       = NULL;
            }

			/* calling hb apis to load glyphs and getting metrics*/ 	
			if(p_renderer->enableshapingeng == SVG_TRUE)
			{
				p_renderer->cached_rendrflags	=	char_info.flags;
				p_renderer->cached_index		=	l_index;
				char_info.textlength			=	textLength;
				
				error = p_renderer->p_grl_shpeng_fp_table->glyphshaping(
													(void*)p_renderer->shared->p_shape_eng,
													(get_char_info_struct*)&char_info,
													(void*)ctx,
													ctx->encoding);
			
				if (error != SVG_NO_ERROR)
				{
					hbGlyphCount = 0;
					grlftSetError(ctx, error);
				}
				else
				{
					hbGlyphCount =	p_renderer->p_grl_shpeng_fp_table->GetGlyphCount(
												(void*)p_renderer->shared->p_shape_eng);
				}
				
				if(!hbGlyphCount) 
				{
					term_found = SVG_TRUE;				
				}
			}

            /* start the main part of this loop. all preparation is now finished */
			loop = 0;

			if((p_renderer->enableshapingeng == SVG_TRUE) && (!hbGlyphCount))
			{
				term_found = SVG_TRUE;				
			}

            while(SVG_FALSE == term_found)
			{
    			/*get the FT bitmap. Also store it in texture in case of GLES. Todo check if kerning is needed*/
                error = local_get_bitmap( ctx, &char_info, loop, cache_bitmap, &add_new_glyph,NULL);
				
                if (error != SVG_NO_ERROR)
                {
                    if (error != SVG_NULL_CHARACTER)
                    {
                        grlftSetError(ctx, error);
						/*end the loop to many different glyphs for cache size*/
						if(error == TOO_MANY_GLYPHS_FOR_CACHE)
						{
							grlftSetError(ctx, TOO_MANY_GLYPHS_FOR_CACHE);
						}
                    }
                    else
                    {
                    	if(p_renderer->enableshapingeng == SVG_FALSE)
                    	{
	                        if(char_info.decodedChar == 0)
	                        {
	                            term_found = SVG_TRUE;
	                        }
	                    }
                    }
                }
                else
                {
                    /* count only drawable glyphs... */
                    glyphs_to_draw++;
                    /* for curved texts calculate next position and/or angle */
                    if(SVG_TRUE == draw_ext_glyphs)
                    {
                        ext_angle = angle[loop];                        
                        /* normalize angle into range of 0 up to 359 */
                        if(ext_angle < MIN_ANGLE)
						{
							ext_angle = MIN_ANGLE;
						}
						if( ext_angle < 0.0f )
						{
							ext_angle += MAX_ANGLE;
						}
						if(ext_angle > MAX_ANGLE)
						{
							ext_angle = MAX_ANGLE;
						}                       
#ifdef HAS_OPENGLES
                        if ((SVG_USE_OPENGLES_1 == p_renderer->drawingAPI) && (NULL == textBBox))
                        {
#ifdef HAS_OPENGLES1
                            glPushMatrix();
                
                            if ((ctx->settings & SVG_FONT_USE_CURRENT_MATRIX) == 0)
                            {
                                glLoadIdentity();
                            }
                
                            /* move to starting position */
                            glTranslatef (position[loop].x, position[loop].y, 0.0f);
                            glRotatef((360.0f - ext_angle), 0.0f, 0.0f, 1.0f );
                            if((p_renderer->scale_x < 1.0) || (p_renderer->scale_y < 1.0))
                            {
                                glScalef(p_renderer->scale_x, p_renderer->scale_y, 1.0);
                            }
                            glPopMatrix();
#else
                            local_matrixIdentity(p_renderer->shared->modelview);
                            /* move to starting position */
                            local_matrixTranslate (p_renderer->shared->modelview,position[loop].x, position[loop].y);
                            local_matrixRotate(p_renderer->shared->modelview, (360.0f - ext_angle));
                            if((p_renderer->scale_x < 1.0) || (p_renderer->scale_y < 1.0))
                            {
                                local_matrixScale(p_renderer->shared->modelview, p_renderer->scale_x, p_renderer->scale_y);
                            }
#endif                            
                        }
#endif                    
#ifdef HAS_OPENVG
                        if ((SVG_USE_OPENVG == p_renderer->drawingAPI) && (NULL == textBBox))
                        {
                            vgLoadMatrix(&matrix[0]);
                                          
                            /* move to starting position */
                            vgTranslate (position[loop].x, position[loop].y);
                                                        vgRotate((360.0f - ext_angle));
                                            
                            if((p_renderer->scale_x < 1.0) || (p_renderer->scale_y < 1.0))
                            {
                                vgScale(p_renderer->scale_x, p_renderer->scale_y);
                                                            }
                        }
#endif        
                        
                        p_renderer->advance_x = 0.0f;
                        p_renderer->advance_y = 0.0f;
                        char_info.next_pos.x = position[loop].x;
                        char_info.next_pos.y = position[loop].y;
                        char_info.angle      = ext_angle;
                        p_renderer->is_first_char = SVG_TRUE;
                        /* draw a single glyph . Texture is always 0*/
                        local_setFontMask(ctx, &char_info);
                    }
                    else
                    {
                        /* calculate cursorposition or bounding box or maxchars */
                        if ((NULL != cursorPos) || (NULL != textBBox) || (SVG_TRUE == draw_text))
                        {
                        	if(p_renderer->enableshapingeng == SVG_FALSE)
                        	{
	                            /* set flag that last char is reached for calc_maximum_extension */
	                            if( (&char_info.theChar[1] == &text[textLength]) ||
									(char_info.theChar[1] == 0))
	                            {
	                                first_or_last_char = SVG_TRUE;
	                            }
	                        }
	                        else
	                        {
	                        	if(loop == (hbGlyphCount-1))
	                        	{
									first_or_last_char = SVG_TRUE;	
	                        	}
	                        }

                            /* do the bbox calculation */
                            local_calc_maximum_extension(ctx, 
                                                         &localTextBBox,
                                                         &cursor_position,
                                                         first_or_last_char);
                    
                            first_or_last_char = SVG_FALSE;
                            /* check if current values exceed maximum values */
                            if((localTextBBox.width  > max_width) || 
                               (localTextBBox.height > max_height))
                            {
                                /* end the loop */
                                term_found = SVG_TRUE;
                            }
                            else
                            {
                                max_chars++;
                            }
                            if(SVG_TRUE == draw_text)
                            {
                                local_setFontMask(ctx, &char_info);
								char_info.next_pos.x = cursor_position.x;
                                char_info.next_pos.y = cursor_position.y;
                                /* store the next position to the given array */
                                if((NULL != cursorPos) && (stored == SVG_FALSE))
                                {
#ifdef HAS_OPENVG
                                    if(SVG_USE_OPENVG == p_renderer->drawingAPI)
                                    {
                                        vgGetMatrix(&matrix_tmp[0]);
                                        cursorPos->x = matrix_tmp[6];
                                        cursorPos->y = matrix_tmp[7];
                                    }
#endif                                    
#ifdef HAS_OPENGLES
                                    if(SVG_USE_OPENGLES_1 == p_renderer->drawingAPI)
                                    {      
#ifdef HAS_OPENGLES1
                                        glGetFloatv(GL_MODELVIEW_MATRIX, &matrix[0]);
                                        cursorPos->x = matrix[12];
                                        cursorPos->y = matrix[13];
#else
                                        cursorPos->x = p_renderer->shared->modelview[3][0];
                                        cursorPos->y = p_renderer->shared->modelview[3][1];
#endif                                        
                                    }
#endif                                    
                                    cursorPos = &cursorPos[1];
                                }
                            }
                        }

                        late_glyph = &p_renderer->shared->late_glyphs[p_renderer->index];
                        late_glyph->cache_current = p_renderer->shared->cache_current;
                        late_glyph->offset_x      = p_renderer->offset_x;
                        late_glyph->offset_y      = p_renderer->offset_y;
                        p_renderer->index++;
                    }
                }
	             
				loop++;

                if(p_renderer->enableshapingeng == SVG_FALSE)
                {                
	                char_info.theChar = &char_info.theChar[1];
	                if(!(char_info.theChar < &text[textLength]))
	                {
	                	term_found = SVG_TRUE;
	                }
	            }
	            else
	            {
	            	if(loop == hbGlyphCount)
	            	{
	                	term_found = SVG_TRUE;
	            	}
	            }
            }
            if ( (l_index == 0) && (add_new_glyph))
            {
                is_update_strk = SVG_TRUE;
            }
            if ( (l_index == 1) && (add_new_glyph))
            {
                is_update_fill = SVG_TRUE;
            }
            stored = SVG_TRUE;
			
#ifdef HAS_OPENGLES1
            if(SVG_USE_OPENGLES_1 == p_renderer->drawingAPI)
            {      
                glPopMatrix();
            }
#endif            
        }
        
		if(p_renderer->enableshapingeng == SVG_TRUE)
		{
			/* destory hb buffer object*/
			p_renderer->p_grl_shpeng_fp_table->DestroyBuffObjects(
							(void*)p_renderer->shared->p_shape_eng);
                     p_renderer->enableshapingeng = SVG_FALSE;
		}
		
		reset_in_use_glyph_flag(p_renderer);
		
		if(glyphs_to_draw == 0)
		{
			draw_text_final = SVG_FALSE;
		}		

		if((ctx->settings & SVG_FONT_NO_ENCLOSING_SPACES) != 0)
        {
			if(SVG_FALSE == p_renderer->is_vertical)
			{
				p_renderer->origin_ver = 0.0f;
				if(SVG_ORIGIN_TOP == ctx->base_line)
				{
					p_renderer->origin_hor = (SVGFloat)-p_renderer->top_most;
				}
				else  if(SVG_ORIGIN_BOTTOM == ctx->base_line)
				{
					p_renderer->origin_hor = (SVGFloat)p_renderer->bottom_most;
				}
				else if( (SVG_ORIGIN_BASELINE == ctx->base_line) )
				{
					p_renderer->origin_hor = (SVGFloat)(p_renderer->stroke_width >> 1) + 1.0f;
				}
			}
			else
			{
				p_renderer->origin_hor = 0.0f;
				if(SVG_ORIGIN_TOP == ctx->base_line)
				{
					p_renderer->origin_ver = -(SVGFloat)p_renderer->left_most;
				}
				else if(SVG_ORIGIN_BOTTOM == ctx->base_line)
				{
					p_renderer->origin_ver = -(SVGFloat)p_renderer->right_most;
				}				
			}
		}
		/* do the final calculation for the bounding box*/
        if(NULL != textBBox)
        {
            /* adjust top left corner in accordance to origin */
            if(SVG_FALSE == p_renderer->is_vertical)
            {
                textBBox->top    = localTextBBox.top + (p_renderer->origin_hor * cos_angle);
                textBBox->left   = localTextBBox.left+ (p_renderer->origin_hor * sin_angle);
            }
            else
            {
                textBBox->top    = localTextBBox.top + (p_renderer->origin_ver * -sin_angle) + (p_renderer->extra_ver * cos_angle);
                textBBox->left   = localTextBBox.left+ (p_renderer->origin_ver * cos_angle) + (p_renderer->extra_ver * sin_angle);
            }
            textBBox->width  = localTextBBox.width;
            textBBox->height = localTextBBox.height;
            textBBox->top   -= textBBox->height;
        }
        else
        {
            /* do the final drawing, when NO_ENCLOSING_SPACES is set and 
               origin is TOP or BOTTOM */
            if( SVG_TRUE == draw_text_final )
            {
                stored    = SVG_FALSE;
                cursorPos = cursorPosArray ;
                p_renderer->index = 0;
#ifdef HAS_OPENVG
                if (SVG_USE_OPENVG == p_renderer->drawingAPI)
                {      
                    for(l_index = p_renderer->start; l_index < p_renderer->end; l_index++)
                    {
                        vgLoadMatrix(&matrix[0]);
                        if(0 == l_index)
                        {
							p_renderer->shared->cache = &p_renderer->shared->cache_strk;
							p_renderer->shared->cache_count = &p_renderer->shared->cache_count_strk;
							p_renderer->shared->hash_tbl_glyphs = p_renderer->shared->hash_tbl_strk;
                            vgSetParameterfv(p_renderer->VGpaint, VG_PAINT_COLOR, 4, p_renderer->stroke_color);
                        }
                        else
                        {
							p_renderer->shared->cache = &p_renderer->shared->cache_fill;
							p_renderer->shared->cache_count = &p_renderer->shared->cache_count_fill;
							p_renderer->shared->hash_tbl_glyphs = p_renderer->shared->hash_tbl_fill;
                            vgSetParameterfv(p_renderer->VGpaint, VG_PAINT_COLOR, 4, p_renderer->fill_color);
                        }
                        vgRotate(360.0f - char_info.angle);
                        vgTranslate( p_renderer->origin_ver, p_renderer->origin_hor);
                        vgRotate( -(360.0f - char_info.angle));
                                    
                        /* move to starting position */
                        vgTranslate (position[0].x, position[0].y);
                        vgRotate(360.0f - char_info.angle);

                        if((p_renderer->scale_x < 1.0) || (p_renderer->scale_y < 1.0))
                        {
                            vgScale(p_renderer->scale_x, p_renderer->scale_y);
                        }

                        for(loop = 0; loop < glyphs_to_draw; loop++)
                        {
                            late_glyph = &p_renderer->shared->late_glyphs[p_renderer->index++];
                            p_renderer->shared->cache_current = late_glyph->cache_current;
                            p_renderer->offset_x = late_glyph->offset_x;
                            p_renderer->offset_y = late_glyph->offset_y;
                            local_render_vg_glyph(ctx);
                            if((NULL != cursorPos) && (stored == SVG_FALSE))
                            {
                                vgGetMatrix(&matrix_tmp[0]);
                                cursorPos->x = matrix_tmp[6];
                                cursorPos->y = matrix_tmp[7];
                                cursorPos = &cursorPos[1];
                            }
                        }
                        stored = SVG_TRUE;
                    }
                }
#endif                
#ifdef HAS_OPENGLES
                if (SVG_USE_OPENGLES_1 == p_renderer->drawingAPI)
                {   
					if (glMultiDrawArraysEXT != NULL)
					{
						drawMultSupported = SVG_TRUE;
						texCoords   = (GLfloat*)GRL_malloc_1D_resource(sizeof(GLfloat)*glyphs_to_draw*8);
						vertx       = (GLfloat*)GRL_malloc_1D_resource(sizeof(GLfloat)*glyphs_to_draw*8);
						first       = (GLint*)  GRL_malloc_1D_resource(sizeof(GLint)  *glyphs_to_draw);
						glyphEdges  = (GLsizei*)GRL_malloc_1D_resource(sizeof(GLsizei)*glyphs_to_draw);
						if( (texCoords == NULL) || (vertx == NULL) || (first == NULL) || (glyphEdges == NULL) )
						{/*not enough memory for all glyphs, but maybe enough to draw glyph by glyph*/
							drawMultSupported = SVG_FALSE;
							grlftSetError(ctx, SVG_OUT_OF_MEMORY);
				            SVG_FNT_F("SVG_OUT_OF_MEMORY IN GRLXXDRAWFONTWITHCURSORINFO ");
						}
					}
                    for(l_index = p_renderer->start; l_index < p_renderer->end; l_index++)
                    {
#ifdef HAS_OPENGLES1
                        if(0 == l_index)
                        {
							glBindTexture(GL_TEXTURE_2D,p_renderer->shared->glWallpaperStrk);
							p_renderer->shared->cache = &p_renderer->shared->cache_strk;
							p_renderer->shared->cache_count = &p_renderer->shared->cache_count_strk;
							p_renderer->shared->hash_tbl_glyphs = p_renderer->shared->hash_tbl_strk;
                            
							glColor4f(p_renderer->stroke_color[0], 
                                      p_renderer->stroke_color[1], 
                                      p_renderer->stroke_color[2], 
                                      p_renderer->stroke_color[3]);
                        }
                        else
                        {
							glBindTexture(GL_TEXTURE_2D,p_renderer->shared->glWallpaperFill);
                            p_renderer->shared->cache = &p_renderer->shared->cache_fill;
							p_renderer->shared->cache_count = &p_renderer->shared->cache_count_fill;
							p_renderer->shared->hash_tbl_glyphs = p_renderer->shared->hash_tbl_fill;
							
							glColor4f(p_renderer->fill_color[0], 
                                      p_renderer->fill_color[1], 
                                      p_renderer->fill_color[2], 
                                      p_renderer->fill_color[3]);
                        }
                        glPushMatrix();
            
                        if ((ctx->settings & SVG_FONT_USE_CURRENT_MATRIX) == 0)
                        {
                            glLoadIdentity();
                        }
                        glRotatef(360.0f - char_info.angle, 0.0f, 0.0f, 1.0f );
                        glTranslatef( p_renderer->origin_ver, p_renderer->origin_hor, 0.0f);
                        glRotatef( -(360.0f - char_info.angle), 0.0f, 0.0f, 1.0f );
            
                        /* move to starting position */
                        glTranslatef (position[0].x, position[0].y, 0.0f);
                        glRotatef(360.0f - char_info.angle, 0.0f, 0.0f, 1.0f );

                        if((p_renderer->scale_x < 1.0) || (p_renderer->scale_y < 1.0))
                        {
                            glScalef(p_renderer->scale_x, p_renderer->scale_y, 1.0);
                        }
#else
                        if(0 == l_index)
                        {
							glBindTexture(GL_TEXTURE_2D,p_renderer->shared->glWallpaperStrk);
							p_renderer->shared->cache = &p_renderer->shared->cache_strk;
							p_renderer->shared->cache_count = &p_renderer->shared->cache_count_strk;
							p_renderer->shared->hash_tbl_glyphs = p_renderer->shared->hash_tbl_strk;

							if ((ctx->settings & SVG_FONT_USE_CURRENT_PROGRAM) == 0)
							{
                                glUniform4fv(p_renderer->shared->colorlocation, 1, &p_renderer->stroke_color[0]);
							}
							if( (is_update_strk) && ((ctx->settings & SVG_FONT_MIPMAPPING) != 0))
							{
								glGenerateMipmap(GL_TEXTURE_2D); 
							}
                        }
                        else
                        {
							glBindTexture(GL_TEXTURE_2D,p_renderer->shared->glWallpaperFill);
                            p_renderer->shared->cache = &p_renderer->shared->cache_fill;
							p_renderer->shared->cache_count = &p_renderer->shared->cache_count_fill;
							p_renderer->shared->hash_tbl_glyphs = p_renderer->shared->hash_tbl_fill;
							
							if ((ctx->settings & SVG_FONT_USE_CURRENT_PROGRAM) == 0)
							{
							    glUniform4fv(p_renderer->shared->colorlocation, 1, &p_renderer->fill_color[0]);
							}
							if( (is_update_fill) && ((ctx->settings & SVG_FONT_MIPMAPPING) != 0))
							{
								glGenerateMipmap(GL_TEXTURE_2D); 
							}
                        }
                        local_matrixIdentity(p_renderer->shared->modelview);
                        local_matrixRotate(p_renderer->shared->modelview, 360.0f - char_info.angle );
                        local_matrixTranslate(p_renderer->shared->modelview, p_renderer->origin_ver, p_renderer->origin_hor);
                        local_matrixRotate(p_renderer->shared->modelview, char_info.angle);
            
                        /* move to starting position */
                        local_matrixTranslate (p_renderer->shared->modelview,position[0].x, position[0].y);
                        local_matrixRotate(p_renderer->shared->modelview, 360.0f - char_info.angle);

                        if((p_renderer->scale_x < 1.0) || (p_renderer->scale_y < 1.0))
                        {
                            local_matrixScale(p_renderer->shared->modelview, p_renderer->scale_x, p_renderer->scale_y);
                        }
#endif
						/*pointers are double checked to avoid qac warnings*/
                        if( (drawMultSupported == SVG_TRUE) &&(texCoords != NULL) && (vertx != NULL) && (first != NULL) && (glyphEdges != NULL) )
						{
							prev_off_x 			= 0;
							prev_off_y 			= 0;
#ifdef HAS_OPENGLES1
                            glGetFloatv(GL_MODELVIEW_MATRIX, &mv_matrix[0]);
#else		
							local_matrixMutiply(p_renderer->shared->mvp, p_renderer->shared->modelview, projection);
							glUniformMatrix4fv(p_renderer->shared->mvplocation, 1, GL_FALSE, &p_renderer->shared->mvp[0][0]);
#endif
							for (loop = 0; loop < glyphs_to_draw; ++loop)
							{
								first[loop] = (GLint)loop*4;
								glyphEdges[loop] = 4;
							}
	                        for(loop = 0; loop < (glyphs_to_draw*8); loop +=8)
	                        {
	                            late_glyph = &p_renderer->shared->late_glyphs[p_renderer->index++];
	                            p_renderer->shared->cache_current = late_glyph->cache_current;
	                            p_renderer->offset_x = late_glyph->offset_x;
	                            p_renderer->offset_y = late_glyph->offset_y;
								vertx[loop]    = prev_off_x + late_glyph->offset_x;
								vertx[loop +1] = prev_off_y + late_glyph->offset_y;
	                            vertx[loop +2] = vertx[loop] + (late_glyph->cache_current->width);
								vertx[loop +3] = vertx[loop +1];
								vertx[loop +4] = vertx[loop +2];
								vertx[loop +5] = vertx[loop +3] + (late_glyph->cache_current->height);
								vertx[loop +7] = vertx[loop +5];
								vertx[loop +6] = vertx[loop];
							
								texCoords[loop ] = late_glyph->cache_current->texture_source_x;
								texCoords[loop +1] = late_glyph->cache_current->texture_source_y;
								texCoords[loop +2] = texCoords[loop] + late_glyph->cache_current->texture_width;
								texCoords[loop +3] = texCoords[loop +1];
								texCoords[loop +4] = texCoords[loop +2];
								texCoords[loop +5] = texCoords[loop +3] + late_glyph->cache_current->texture_height;
								texCoords[loop +6] = texCoords[loop];
								texCoords[loop +7] = texCoords[loop +5];
								prev_off_x = vertx[loop];
								prev_off_y = vertx[loop+1];

	                            if((NULL != cursorPos) && (stored == SVG_FALSE))
	                            {
#ifdef HAS_OPENGLES1
									glTranslatef (p_renderer->offset_x, p_renderer->offset_y, 0.0f);
	                                glGetFloatv(GL_MODELVIEW_MATRIX, &matrix[0]);
	                                cursorPos->x = matrix[12];
	                                cursorPos->y = matrix[13];
#else
									local_matrixTranslate(p_renderer->shared->modelview, p_renderer->offset_x, p_renderer->offset_y);
	                                cursorPos->x = p_renderer->shared->modelview[3][0];
	                                cursorPos->y = p_renderer->shared->modelview[3][1];
#endif                                
	                                cursorPos = &cursorPos[1];
	                            }
	                        }
#ifdef HAS_OPENGLES1
							glTexCoordPointer(2, GL_FLOAT, 0, texCoords);
							glVertexPointer(2, GL_FLOAT, 0, vertx);
							glLoadMatrixf(mv_matrix);
#else
							if ((ctx->settings & SVG_FONT_USE_CURRENT_PROGRAM) == 0)
							{
								glVertexAttribPointer(p_renderer->shared->texturelocation, 2, GL_FLOAT, 0, 0, texCoords);
								glVertexAttribPointer(p_renderer->shared->vertexlocation, 2, GL_FLOAT, 0, 0, vertx);
							} else {
								glGetIntegerv(GL_CURRENT_PROGRAM, &current_program);
								glUniformMatrix4fv(glGetUniformLocation((SVGUint32)current_program, "mvp"),
										1, GL_FALSE, &p_renderer->shared->mvp[0][0]);
								glVertexAttribPointer((SVGUint32)glGetAttribLocation((SVGUint32)current_program, "inputtexcoord"),
										2, GL_FLOAT, 0, 0, texCoords);
								glVertexAttribPointer((SVGUint32)glGetAttribLocation((SVGUint32)current_program, "position"),
										2, GL_FLOAT, 0, 0, vertx);
							}
#endif
							glMultiDrawArraysEXT(GL_TRIANGLE_FAN, first, glyphEdges,(SVGInt32)glyphs_to_draw);
						}
						else
						{
							for(loop = 0; loop < glyphs_to_draw; loop++)
                       		{
	                            late_glyph = &p_renderer->shared->late_glyphs[p_renderer->index++];
	                            p_renderer->shared->cache_current = late_glyph->cache_current;
	                            p_renderer->offset_x = late_glyph->offset_x;
	                            p_renderer->offset_y = late_glyph->offset_y;
							
	                            local_render_gl_glyph(ctx);
	                            if((NULL != cursorPos) && (stored == SVG_FALSE))
	                            {
#ifdef HAS_OPENGLES1
	                                glGetFloatv(GL_MODELVIEW_MATRIX, &matrix[0]);
	                                cursorPos->x = matrix[12];
	                                cursorPos->y = matrix[13];
#else
	                                cursorPos->x = p_renderer->shared->modelview[3][0];
	                                cursorPos->y = p_renderer->shared->modelview[3][1];
#endif                                
	                                cursorPos = &cursorPos[1];
	                            }
                        	}
						}
                        stored = SVG_TRUE;
#ifdef HAS_OPENGLES1
                        if(SVG_USE_OPENGLES_1 == p_renderer->drawingAPI)
						{
							glPopMatrix();
						}
#endif                        
                    }
                }
#endif            
            }
        }
		
/* reset context values */
#ifdef HAS_OPENGLES
		if(texCoords)
			GRL_free_1D_resource(texCoords);
		if(vertx)
			GRL_free_1D_resource(vertx);
		if(first)
			GRL_free_1D_resource(first);
		if(glyphEdges)
			GRL_free_1D_resource(glyphEdges);
        if ((SVG_USE_OPENGLES_1 == p_renderer->drawingAPI) && (NULL == textBBox))
        {      

            error_api = glGetError();
            if(GL_NO_ERROR != error_api)
            {
			    SVG_FNT_E("SVG_OPENGL_ERROR %d IN "
			    		"GRLXXDRAWFONTWITHCURSORINFO ",error_api);
            }
#ifdef HAS_OPENGLES1
            glColor4f(current_color[0], 
                      current_color[1], 
                      current_color[2], 
                      current_color[3]);
            glDisable(GL_BLEND);
            glColorMask(GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE);
          	glTexEnvx(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
          	glDisable(GL_TEXTURE_2D);
#else
			if ((ctx->settings & SVG_FONT_USE_CURRENT_PROGRAM) == 0)
			{
            	glUseProgram((SVGUint32)current_program);
            }
#endif
        }
#endif        
#ifdef HAS_OPENVG
        if ((SVG_USE_OPENVG == p_renderer->drawingAPI) && (NULL == textBBox))
        {      
            error_api = vgGetError();
            if(VG_NO_ERROR != error_api)
            {
			    SVG_FNT_E("SVG_OPENVG_ERROR %d IN "
			    		"GRLXXDRAWFONTWITHCURSORINFO ",error_api);
            }
            vgSetParameterfv(p_renderer->VGpaint, VG_PAINT_COLOR, 4, current_color);
            vgLoadMatrix(&vg_master_matrix[0]);
        }
#endif        

        p_renderer->scale_x = 1.0f;
        p_renderer->scale_y = 1.0f;
        ctx->letter_distance = distance;

    }
	
    return max_chars;
}

#ifdef HAS_OPENVG
void local_render_vg_glyph(const SVGFontContext* ctx)
{
	SVGFontFreeTypeRenderer   *p_renderer  = (SVGFontFreeTypeRenderer*)
	                                                ctx->p_fontRendererContext;
    grl_internal_cache_struct* current = p_renderer->shared->cache_current;
    vgTranslate (p_renderer->offset_x, p_renderer->offset_y);
    if(0 != (current->width * current->height))
    {   
        vgDrawImage(current->VGimage);
    }
}
#endif //HAS_OPENVG

#ifdef HAS_OPENGLES
static void local_render_gl_glyph(const SVGFontContext* ctx)
{
    GLfloat                    texCoords[8] = {0.0f};
    GLfloat                    vertx[8]    = {0.0f};
    SVGFontFreeTypeRenderer   *p_renderer         = (SVGFontFreeTypeRenderer*)
                                                        ctx->p_fontRendererContext;

    grl_internal_cache_struct* current      = p_renderer->shared->cache_current;

#ifdef HAS_OPENGLES1
    glTranslatef (p_renderer->offset_x, p_renderer->offset_y, 0.0f);
#else
    local_matrixTranslate(p_renderer->shared->modelview, p_renderer->offset_x, p_renderer->offset_y);
    local_matrixMutiply(p_renderer->shared->mvp, p_renderer->shared->modelview, projection);
    glUniformMatrix4fv(p_renderer->shared->mvplocation, 1, GL_FALSE, &p_renderer->shared->mvp[0][0]);
#endif    
    /* destination vertx */
    /* based on 0 because translation is done by OpenGL */
    if(0 != (current->width * current->height))
    {   
        vertx[2] = (current->width);
        vertx[4] = vertx[2];
        vertx[5] = (current->height);
        vertx[7] = vertx[5];
    
        /* source vertx */
        texCoords[0] = current->texture_source_x;
        texCoords[1] = current->texture_source_y;
        texCoords[2] = texCoords[0] + current->texture_width;
        texCoords[3] = texCoords[1];
        texCoords[4] = texCoords[2];
        texCoords[5] = texCoords[3] + current->texture_height;
        texCoords[6] = texCoords[0];
        texCoords[7] = texCoords[5];
    
#ifdef HAS_OPENGLES1
        glTexCoordPointer(2, GL_FLOAT, 0, texCoords);
        glVertexPointer(2, GL_FLOAT, 0, vertx);
#else
        if ((ctx->settings & SVG_FONT_USE_CURRENT_PROGRAM) == 0)
        {
            glVertexAttribPointer(p_renderer->shared->texturelocation, 2, GL_FLOAT, 0, 0, texCoords);
            glVertexAttribPointer(p_renderer->shared->vertexlocation, 2, GL_FLOAT, 0, 0, vertx);
        } else {
            SVGInt32 current_program = 0;
            glGetIntegerv(GL_CURRENT_PROGRAM, &current_program);
            glUniformMatrix4fv(glGetUniformLocation((SVGUint32)current_program, "mvp"),
            		1, GL_FALSE, &p_renderer->shared->mvp[0][0]);
            glVertexAttribPointer((SVGUint32)glGetAttribLocation((SVGUint32)current_program, "inputtexcoord"),
        		    2, GL_FLOAT, 0, 0, texCoords);
            glVertexAttribPointer((SVGUint32)glGetAttribLocation((SVGUint32)current_program, "position"),
        		    2, GL_FLOAT, 0, 0, vertx);
        }
#endif        
        glDrawArrays(GL_TRIANGLE_FAN, 0, 4);
    }
}

static void local_store_gl_bitmap(const SVGFontContext* ctx,
                           get_char_info_struct* char_info,
                           SVGUint32 l_index)
{
	SVGFontFreeTypeRenderer   *p_renderer         = (SVGFontFreeTypeRenderer*)
                                                    ctx->p_fontRendererContext;
 	FT_Glyph                  glyph               = {0};
 	FT_GlyphSlot              slot                = NULL;
 	SVGUint32 				  loop 				  = 0;
 	SVGInt32 				  top 				  = 0;
 	SVGInt32 				  left 				  = 0;
	
 	SVGUint8*   dist_buffer                       = NULL;
    SVGInt32    glyphAlignment = p_renderer->glUnpackAlignment;
 	SVGUint32   aligned_glyph_width               = 0;

	if(0 == FT_Load_Glyph( p_renderer->shared->face,
                           char_info->theGlyph,
                           (SVGInt32)char_info->flags ))
    {
    	FT_Get_Glyph( p_renderer->shared->face->glyph, &glyph );
    	/*use stroker or just get the slot*/
    	if (0 == l_index)
    	{
            FT_Glyph_Stroke( &glyph, p_renderer->shared->stroker, TRUE );
            if (0 == FT_Glyph_To_Bitmap(&glyph, 
                                       FT_RENDER_MODE_NORMAL,
                                       0, 
                                       1))
            {
            	slot 		 = p_renderer->shared->face->glyph; 
            	slot->bitmap = ((FT_BitmapGlyph)(glyph))->bitmap;
            	top			 = (SVGInt32)((FT_BitmapGlyph)(glyph))->top;
            	left    	 = (SVGInt32)((FT_BitmapGlyph)(glyph))->left;
				if(left < 0)
				{
					left = 0;
				}
            }
        } 
        else
        {
	    	/* do the rendering and setup for "filled" chars */
	        if (0 == FT_Glyph_To_Bitmap(&glyph, 
                                       FT_RENDER_MODE_NORMAL,
                                       0, 
                                       1))
            {
            	slot 		 = p_renderer->shared->face->glyph; 
            	slot->bitmap = ((FT_BitmapGlyph)(glyph))->bitmap;
            	top			 = (SVGInt32)((FT_BitmapGlyph)(glyph))->top;
            	left    	 = (SVGInt32)((FT_BitmapGlyph)(glyph))->left;
			}
        }
        	
    	if ((NULL != slot) && (NULL != slot->bitmap.buffer))
		{
            aligned_glyph_width = (SVGUint32)slot->bitmap.width + (SVGUint32)glyphAlignment - ((SVGUint32)slot->bitmap.width % (SVGUint32)glyphAlignment);

            dist_buffer = (SVGUint8*)GRL_malloc_1D_resource( (SVGUint32)slot->bitmap.rows*aligned_glyph_width*(sizeof(SVGUint8)));
            if(dist_buffer != NULL)
            {
                memset( dist_buffer, 0x0, (SVGUint32)slot->bitmap.rows*aligned_glyph_width);
                for(loop = 0; loop < (SVGUint32)slot->bitmap.rows; loop++)
                {
                    memcpy( &dist_buffer[ ((SVGUint32)(slot->bitmap.rows - 1) - loop)*aligned_glyph_width] ,
                            &slot->bitmap.buffer[loop * (SVGUint32)slot->bitmap.width],
                            (SVGUint32)slot->bitmap.width);
                }
                glTexSubImage2D(GL_TEXTURE_2D,
                                    0,
                                    (GLint)(char_info->next_pos.x + left ),
                                    (GLint)(char_info->next_pos.y + (p_renderer->origin_hor + top - slot->bitmap.rows)),
                                    (GLsizei)aligned_glyph_width,
                                    (GLsizei)slot->bitmap.rows,
                                    GL_ALPHA,
                                    GL_UNSIGNED_BYTE,
                                    dist_buffer);
                GRL_free_1D_resource(dist_buffer);
            }
            else
            {
			    SVG_FNT_F("SVG_OUT_OF_MEMORY IN "
			    		"GRLXXDRAWFONTWITHCURSORINFO");
            }
			
		}
		FT_Done_Glyph(glyph);                    	    
	}
} 
#endif //HAS_OPENGLES1


SVGError local_add_wallpaper_glyph_internal(SVGFontFreeTypeRenderer*   p_renderer,
                               grl_internal_cache_struct* glyph,
                               SVGUint8*                  buffer)
{
	SVGError	err		= SVG_NO_ERROR;
    textureRow* row     = p_renderer->row;
    textureCol* col     = NULL;
	textureCol* pre_col	= NULL;
    textureRow* new_row = NULL;
    textureCol* new_col = NULL;
    SVGBoolean  added   = SVG_FALSE;
	
	SVGUint8*	dist_buffer = NULL;
    SVGInt32    glyphAlignment = p_renderer->glUnpackAlignment;
	SVGUint32   padding = (SVGUint32)glyph->width % (SVGUint32)glyphAlignment;
	SVGUint32	aligned_glyph_width = (SVGUint32)glyph->width + (SVGUint32)glyphAlignment - padding;
#ifdef HAS_OPENVG
	VGImage vgWallpaper = VG_INVALID_HANDLE;
#endif

#ifdef HAS_OPENGLES
    SVGUint32   loop    = 0;

    if ( (p_renderer->shared->mem_guard1 != 0xdeadbeef) ||
         (p_renderer->shared->mem_guard2 != 0xdeadbeef)  )
    {
        FILE *err_mem;

        err_mem = fopen("/dev/errmem","wa");
     
	    SVG_FNT_E("VG_IMAGE_FAIL IN GRLXXDRAWFONTWITHCURSORINFO");
        if (err_mem)
        {
            fprintf(err_mem,"ERROR MEMORY CORRUPTION IN SVGFONTLIB\n");
            fflush(err_mem);
            fprintf(err_mem,"ERROR MEMORY CORRUPTION IN SVGFONTLIB: strk_count: %d, fill_count: %d, index: %d"
                    "p_renderer->shared->glRowStrk: %p, p_renderer->shared->glRowFill: %p \n",
                    p_renderer->shared->cache_count_strk,
                    p_renderer->shared->cache_count_fill,
                    glyph->index,
                    p_renderer->shared->glRowStrk,
                    p_renderer->shared->glRowFill);
            fclose(err_mem);
        }
    }
#endif    
    if(NULL == row)
    {
        /* when "nothing" exists create first row and col */
        new_row = (textureRow*)GRL_malloc_1D_resource(sizeof(textureRow));
        new_col = (textureCol*)GRL_malloc_1D_resource(sizeof(textureCol));
        if ((new_row == NULL) || (new_col == NULL))
        {
        	GRL_free_1D_resource((void *)new_row);
        	GRL_free_1D_resource((void *)new_col);
		    SVG_FNT_F("SVG_OUT_OF_MEMORY IN "
		    		"GRLXXDRAWFONTWITHCURSORINFO");
        	err = SVG_OUT_OF_MEMORY;
        }
        else
	    {
	        new_col->x = 0.0f;
	        new_col->next = NULL;
	        new_col->free_width = p_renderer->wallpaperWidth;
	
	        new_row->y = 0.0f;
	        new_row->height = 0.0f;
	        new_row->col = new_col;
	        new_row->next = NULL;
			new_row->prev = NULL;
	        new_row->fixed = SVG_FALSE;
	        row = new_row;
	        p_renderer->row = row;
#ifdef HAS_OPENVG
	        if (SVG_USE_OPENVG == p_renderer->drawingAPI)
	        {
				if( glyph->index == 0)
					p_renderer->shared->vgRowStrk = row;
				else
					p_renderer->shared->vgRowFill = row;
	        }
#endif        
#ifdef HAS_OPENGLES
	        if (SVG_USE_OPENGLES_1 == p_renderer->drawingAPI)
	        {
				if( glyph->index == 0)
					p_renderer->shared->glRowStrk = row;
				else
					p_renderer->shared->glRowFill = row;
	        }
#endif        
    	}
    }
    /* add glyph only when the row is not fixed( no next row exits)
       or when height is less then the row height */
    while((NULL != row) && (SVG_FALSE == added) && (err == SVG_NO_ERROR))
    {
        col = row->col;
        while((NULL != col) && (SVG_FALSE == added)&&
          ((row->fixed == SVG_FALSE) || (glyph->height <= row->height)))
        {
            if(col->free_width >= aligned_glyph_width)
            {
                /* add glyph when free entry is big enough */
                glyph->source_x = col->x;
                glyph->source_y = row->y;
                col->free_width -= aligned_glyph_width;
                col->x          += aligned_glyph_width;
                added = SVG_TRUE;
				/*remove column if free size is null
				* but only if element doesn't match exacly the end of the column*/
				if( (col->free_width == 0) && ( col->next != NULL) )
				{
					if( pre_col == NULL )
					{
						row->col = col->next;
					}
					else
					{
						pre_col->next = col->next;
					}
					GRL_free_1D_resource((void*)col);
				}
                /* remember the tallest glyph within this row */
                if(glyph->height > row->height)
                {
                    row->height = glyph->height;
                }				
#ifdef HAS_OPENVG
                if (SVG_USE_OPENVG == p_renderer->drawingAPI)
                {
					if( glyph->index == 0 )
						vgWallpaper = p_renderer->shared->vgWallpaperStrk;
					else
						vgWallpaper = p_renderer->shared->vgWallpaperFill;
                    /* copy font bitmap to VG wallpaper texture memory */  
                    vgImageSubData ( vgWallpaper,
                                    &buffer[(SVGUint32)(glyph->width * (glyph->height - 1))],
                                    -((SVGInt32)(glyph->width)),
                                    VG_A_8,
                                    (SVGInt32)glyph->source_x,
                                    (SVGInt32)glyph->source_y,
                                    (SVGInt32)glyph->width,
                                    (SVGInt32)glyph->height);
                    glyph->VGimage = vgChildImage( vgWallpaper,
                                                  (SVGInt32)glyph->source_x,
                                                  (SVGInt32)glyph->source_y,
                                                  (SVGInt32)glyph->width,
                                                  (SVGInt32)glyph->height);
                    if(0 == glyph->VGimage)
                    {
            		    SVG_FNT_E("VG_IMAGE_FAIL IN GRLXXDRAWFONTWITHCURSORINFO");
                        err = VG_IMAGE_FAIL;
                    }

                    glyph->texture_width    = glyph->width;
                    glyph->texture_height   = glyph->height;
                    glyph->texture_source_x = glyph->source_x;
                    glyph->texture_source_y = glyph->source_y;
                }
#endif                          
#ifdef HAS_OPENGLES
                if (SVG_USE_OPENGLES_1 == p_renderer->drawingAPI)
                { 	
					dist_buffer = (SVGUint8*)GRL_malloc_1D_resource( (SVGUint32)glyph->height*aligned_glyph_width*(sizeof(SVGUint8)));
					if(dist_buffer != NULL)
					{
						glyph->texture_width    = (glyph->width ) / 
												  (SVGFloat)(p_renderer->wallpaperWidth);
						glyph->texture_height   = (glyph->height ) / 
												  (SVGFloat)(p_renderer->wallpaperHeight);

						glyph->texture_source_x = (glyph->source_x) / 
												  (SVGFloat)(p_renderer->wallpaperWidth);
						glyph->texture_source_y = (glyph->source_y) / 
												  (SVGFloat)(p_renderer->wallpaperHeight);
						/*swap glyph verticaly due different coord-systems*/
						memset( dist_buffer, 0x0, (SVGUint32)(glyph->height*aligned_glyph_width)); 
						for(loop = 0; loop < glyph->height; loop++)
						{
							memcpy( &dist_buffer[ ((SVGUint32)(glyph->height - 1) - loop)*aligned_glyph_width] , 
									&buffer[loop * (SVGUint32)glyph->width],
									(SVGUint32)glyph->width);
						}
						glTexSubImage2D(GL_TEXTURE_2D,
											0,
											(GLint)glyph->source_x,
											(GLint)glyph->source_y,
											(GLsizei)aligned_glyph_width,
											(GLsizei)glyph->height,
											GL_ALPHA, 
											GL_UNSIGNED_BYTE,
											dist_buffer);
						GRL_free_1D_resource(dist_buffer);
						
					}
					else
					{
            		    SVG_FNT_F("SVG_OUT_OF_MEMORY IN "
            		    		"GRLXXDRAWFONTWITHCURSORINFO");
						err = SVG_OUT_OF_MEMORY;
					}
				}
#endif
            }
            else
            {
				pre_col = col;
                /* goto next free element within row */
                col = col->next;
				
            }
        }      
        if((SVG_FALSE == added) && (NULL == row->next) && (err == SVG_NO_ERROR))
        {
            if((row->y + row->height + glyph->height) < p_renderer->wallpaperHeight)
            {
                /* when current row is full or glyph does not  add a new one */
                new_row = (textureRow*)GRL_malloc_1D_resource(sizeof(textureRow));
                new_col = (textureCol*)GRL_malloc_1D_resource(sizeof(textureCol));
                if ((new_row == NULL) || (new_col == NULL))
		        {
		        	GRL_free_1D_resource((void *)new_row);
		        	GRL_free_1D_resource((void *)new_col);
        		    SVG_FNT_F("SVG_OUT_OF_MEMORY IN GRLXXDRAWFONTWITHCURSORINFO");
		        	err = SVG_OUT_OF_MEMORY;
		        }
		        else
		        {
	                new_col->x = 0;
	                new_col->free_width = p_renderer->wallpaperWidth;
	                new_col->next = NULL;
	    
	                row->fixed = SVG_TRUE;
	                new_row->fixed = SVG_FALSE;
	                new_row->y = row->y + (row->height + texture_Ygap);
	                new_row->height = 0;
	                new_row->col = new_col;
	                new_row->next = NULL;
                    new_row->prev = row;
	                row->next = new_row;
	            }
            }
            else
            {
                sprintf((SVGChar*)&prn_buf[0],"fail add glyph with height%f and width%f. cached glyphs:%d",glyph->height,glyph->width,*p_renderer->shared->cache_count);
    		    SVG_FNT_D("%s",&prn_buf[0]);
                err = FAIL_ADD_WALLPAPER;
            }
        }
		pre_col = NULL;
        row = row->next;
    }
    if(SVG_FALSE == added)
    {
	    SVG_FNT_E("FAIL_ADD_WALLPAPER IN GRLXXDRAWFONTWITHCURSORINFO");
        err = FAIL_ADD_WALLPAPER;          
    }
    return err;
}


static SVGError local_cleanup_cache(SVGFontFreeTypeRenderer*   p_renderer)
{
	SVGUint32                 	removed_glyphs     = 0;
	SVGUint32                   used_glyphs        = 0;
	SVGError                  	ret                = SVG_NO_ERROR;
	UTIL_hash_elem_t	  		entry_to_find	   = {NULL, 0, NULL};
	grl_internal_cache_struct 	*cache_current     = NULL;
	grl_internal_cache_struct 	*cache_current_del = NULL;
	/* cleaning up 'cache_cleanup_ratio' percent of currently used cache */
	SVGUint32		   			cache_cleaned_size = *p_renderer->shared->cache_count
														- (SVGUint32)(*p_renderer->shared->cache_count*((SVGFloat)cache_cleanup_ratio/100.0f));
	SVGUint32		   			cnt 		       = 0;

	/*hash table is full */
    SVG_FNT_W("Cache is full. %d percent of cache will be cleaned."
    		"Adapt SVGFNTCACHECNT to increase performance"
    		,cache_cleanup_ratio);

	cache_current 	= *p_renderer->shared->cache;
	cnt 			= 0;
	while( (cache_current != NULL) &&
			(cnt < cache_cleaned_size) )
	{
		cache_current = cache_current->next;
		cnt++;
	}

	while((NULL != cache_current) && (ret == SVG_NO_ERROR))
	{
		/* throw away glyph only when it is not in use! */
		if( cache_current->is_in_use == SVG_FALSE)
		{
			entry_to_find.key = cache_current;
			entry_to_find.key_len = sizeof(SVGUint32)*3;
			entry_to_find.value = cache_current;

			/* remove the glyph from the hash table */
			ret = UTIL_rem_hsh(p_renderer->shared->hash_tbl_glyphs,&entry_to_find);
			if( ret == SVG_NO_ERROR)
			{
#ifdef HAS_OPENVG
				if(SVG_USE_OPENVG == p_renderer->drawingAPI)
				{
					if( 0 != cache_current->VGimage)
					{
						vgDestroyImage(cache_current->VGimage);
					}
				}
#endif
				if (cache_current->glyph_done != SVG_TRUE)
				{
					FT_Done_Glyph(cache_current->ft_glyph);
				}
				/*remove the entry from wallpaper */
				if( cache_current->cached_bitmap == SVG_TRUE)
				{
					local_del_wallpaper_glyph( p_renderer, cache_current );
				}

				/* remove entry from cache chain */
				if(cache_current == *p_renderer->shared->cache)
				{
					*p_renderer->shared->cache = cache_current->next;
				}
				else
				{
					cache_current->prev->next = cache_current->next;
				}
				if( cache_current->next != NULL)
				{
					cache_current->next->prev = cache_current->prev;
				}
				removed_glyphs++;

				/* remove resource */
				(*p_renderer->shared->cache_count)--;
				cache_current_del = cache_current;
				cache_current = cache_current->next;
				GRL_free_1D_resource((void*)cache_current_del);
			}
			else
			{
				/*error by removing hash value, hash table corrupted???*/
				ret = HASHENTRY_REMOVE_ERROR;
			    SVG_FNT_E("Hash element could not be removed it's can "
			    		"leads to cache inconsistency and memory leaks");
			}
		}
		else
		{
			used_glyphs++;
			cache_current = cache_current->next;
		}
	}
	sprintf((SVGChar*)&prn_buf[0],"removed glyphs:%d used glyphs:%d chached_glyphs : %d",removed_glyphs, used_glyphs, *p_renderer->shared->cache_count);
    SVG_FNT_D("%s",&prn_buf[0]);
	if( 0 == removed_glyphs )
	{
		ret = TOO_MANY_GLYPHS_FOR_CACHE;
	    SVG_FNT_E("TOO_MANY_GLYPHS_FOR_CACHE IN GRLXXDRAWFONTWITHCURSORINFO");
	}
	return ret;
}

static SVGError local_add_wallpaper_glyph(SVGFontFreeTypeRenderer*   p_renderer,
		grl_internal_cache_struct* glyph,
		SVGUint8*                  buffer)
{
	SVGError err = SVG_NO_ERROR;
	SVGUint32 retry = 0;
	/* retry mechanism when adding to texture fails due to any reason */
	while(SVG_NO_ERROR != local_add_wallpaper_glyph_internal(p_renderer, glyph, buffer) && (retry < max_retries))
	{
		retry++;
		/* free up cache elements */
		if(TOO_MANY_GLYPHS_FOR_CACHE ==local_cleanup_cache(p_renderer))
		{
			/* abort retry loop when no glyph could be freed */
			retry = max_retries;
		}
		/* now finally nothing could be added to cache so throw an error */
		if(retry == max_retries)
		{
			err = TOO_MANY_GLYPHS_FOR_CACHE;
		}
	}

	return err;
}


static void local_del_wallpaper_glyph(SVGFontFreeTypeRenderer*   p_renderer,
                               grl_internal_cache_struct* glyph)
{
	textureRow* row     = NULL;
    textureCol* col     = NULL;
    textureCol* new_col = NULL;
    textureCol* nextCol = NULL;
    SVGBoolean  deleted = SVG_FALSE;
    SVGUint8*  Resource = NULL;

    SVGInt32    glyphAlignment = 0;
    SVGUint32   padding = 0;
    SVGUint32   aligned_glyph_width= 0;
#ifdef HAS_OPENVG
	VGImage vgWallpaper = VG_INVALID_HANDLE;

	if (SVG_USE_OPENVG == p_renderer->drawingAPI)
	{
		if( glyph->index == 0)
		{
			p_renderer->row = p_renderer->shared->vgRowStrk;
			vgWallpaper = p_renderer->shared->vgWallpaperStrk;
		}
		else
		{
			p_renderer->row = p_renderer->shared->vgRowFill;
			vgWallpaper = p_renderer->shared->vgWallpaperFill;
		}
	}
#endif        
#ifdef HAS_OPENGLES

	glyphAlignment = p_renderer->glUnpackAlignment;
    padding = (SVGUint32)glyph->width % (SVGUint32)glyphAlignment;
    aligned_glyph_width = (SVGUint32)glyph->width + (SVGUint32)glyphAlignment - padding;

    if (SVG_USE_OPENGLES_1 == p_renderer->drawingAPI)
	{
		if( glyph->index == 0)
		{
			p_renderer->row = p_renderer->shared->glRowStrk;
			glBindTexture(GL_TEXTURE_2D, p_renderer->shared->glWallpaperStrk);
		}
		else
		{
			p_renderer->row = p_renderer->shared->glRowFill;
			glBindTexture(GL_TEXTURE_2D, p_renderer->shared->glWallpaperFill);
		}
	}
#endif 
	row = p_renderer->row;
    while((NULL != row) && (SVG_FALSE == deleted))
    {
        if (row->y == glyph->source_y)
        {
#ifdef HAS_OPENVG
			if (SVG_USE_OPENVG == p_renderer->drawingAPI)
			{
				Resource = (SVGUint8*)GRL_malloc_1D_resource((SVGUint32)(row->height * glyph->width));
				if( NULL != Resource)
				{
					memset(Resource, 0x00, (SVGUint32)(row->height * glyph->width));
					vgImageSubData ( vgWallpaper,
									Resource,
									(SVGInt32)(glyph->width),
									VG_A_8,
									(SVGInt32)glyph->source_x,
									(SVGInt32)glyph->source_y,
									(SVGInt32)glyph->width,
									(SVGInt32)row->height);
					GRL_free_1D_resource((void*)Resource);
				}
				else
				{
				    SVG_FNT_W("Wallpaper not cleared");
				}
			}
#endif
#ifdef HAS_OPENGLES		
			/*clear the texture region*/
			if (SVG_USE_OPENGLES_1 == p_renderer->drawingAPI)
			{
				Resource = (SVGUint8*)GRL_malloc_1D_resource((SVGUint32)(row->height * aligned_glyph_width));
				if( NULL != Resource)
				{
					memset(Resource, 0x00, (SVGUint32)(row->height * aligned_glyph_width));
					glTexSubImage2D(GL_TEXTURE_2D, 
							 0, 
							 (GLint)glyph->source_x,
							 (GLint)glyph->source_y,
							 (GLsizei) aligned_glyph_width,
							 (GLsizei) row->height, 
							 GL_ALPHA,
							 GL_UNSIGNED_BYTE,
							 Resource);
					GRL_free_1D_resource((void*)Resource);
				}
				else
				{
				    SVG_FNT_W("Wallpaper not cleared");
				}
			}
#endif
            col = row->col;
            while(SVG_FALSE == deleted)
            {
                /* add the new free element depending on x position 
                   before or after then previous element */
                
				if( (NULL != col->next) && (col->next->x < glyph->source_x) /*&& (col->x < glyph->source_x)*/)
                {
                    col = col->next;
                }
                else
                {
                    new_col = (textureCol*)GRL_malloc_1D_resource(sizeof(textureCol));
                     new_col->free_width = aligned_glyph_width;
                    new_col->x = glyph->source_x;
                    sprintf((SVGChar*)&prn_buf[0],"new free element@ :x: %f; y:%f; width:%f\n", new_col->x, row->y,new_col->free_width);
				    SVG_FNT_D("%s",&prn_buf[0]);
                    if( (col == row->col) && (col->x > glyph->source_x) )
                    {
                        new_col->next = col;
                        row->col = new_col;
                    }
                    else
                    {
                        new_col->next = col->next;
                        col->next = new_col;
                    }
                    deleted = SVG_TRUE;
                    /* now do the defragmentation within row */
                    col = row->col;
                    while(NULL != col)
                    {
                        if(NULL != col->next)
                        {
                            /* +1 is a tentative solution. currently not clear why needed */
                            if((col->x + col->free_width ) == col->next->x)
                            {
                                sprintf((SVGChar*)&prn_buf[0],"removing element:x: %f; y:%f; width:%f\n", col->next->x, row->y,col->next->free_width);
            				    SVG_FNT_D("%s",&prn_buf[0]);
                                nextCol = col->next;
                                col->next = nextCol->next;
                                col->free_width += nextCol->free_width;
                                sprintf((SVGChar*)&prn_buf[0],"modified element:x: %f; y:%f; width:%f\n", col->x, row->y,col->free_width);
            				    SVG_FNT_D("%s",&prn_buf[0]);
                                GRL_free_1D_resource((void*)nextCol);
                            }
                            else
                            {
                                /* go only to next element when current is not
                                   "combinable" */
                                col = col->next;
                            }
                        }
                        else
                        {
                            /* will end up the loop */
                            col = col->next;
                        }
                    }
 					/* preventing "empty unused rows" */

					if((NULL == row->next) && (p_renderer->wallpaperWidth == row->col->free_width))
					{
					    SVG_FNT_D("removing empty row\n");
						if(row->prev == NULL)
						{
							/* it is the first row */
#ifdef HAS_OPENVG
							if (SVG_USE_OPENVG == p_renderer->drawingAPI)
							{
								if( glyph->index == 0)
								{
									p_renderer->shared->vgRowStrk = NULL;
								}
								else
								{
									p_renderer->shared->vgRowFill = NULL;
								}
							}
#endif
#ifdef HAS_OPENGLES
							if (SVG_USE_OPENGLES_1 == p_renderer->drawingAPI)
							{
								if( glyph->index == 0)
								{
									p_renderer->shared->glRowStrk = NULL;
								}
								else
								{
									p_renderer->shared->glRowFill = NULL;
								}
							}
#endif
							p_renderer->row = NULL;
						}
						else
						{
							row->prev->next = NULL;
						}
						GRL_free_1D_resource((void*)row->col);
						GRL_free_1D_resource((void*)row);
						row = NULL;
					}
				}
			}
		}
		if(NULL != row)
		{
			row = row->next;
		}
	}
}


static SVGBoolean local_create_shared_resources(SVGFontFreeTypeRenderer* p_renderer, SVGUint32 textLength)
{
    SVGBoolean              ret                = SVG_FALSE;

    if(NULL == p_renderer->shared->late_glyphs)
    {
        p_renderer->shared->late_glyphs = (grl_late_draw_struct*)GRL_malloc_1D_resource(sizeof(grl_late_draw_struct) * 2 * textLength);
        if (p_renderer->shared->late_glyphs == NULL) {
            ret = SVG_TRUE;
        } else {
            p_renderer->shared->late_glyphs_size = 2 * textLength;
        }
    }
    
    return ret;
}

static void local_destroy_shared_resources(SVGFontFreeTypeRenderer* p_renderer)
{
    if( NULL != p_renderer->shared->late_glyphs)
    {
        GRL_free_1D_resource((void*)p_renderer->shared->late_glyphs);
    }
    p_renderer->shared->late_glyphs = NULL;

    p_renderer->shared->late_glyphs_size = 0xffffffff;
}

static BOOL local_hash_cmp_fn(UTIL_hash_elem_t* e1, UTIL_hash_elem_t* e2)
{
	BOOL r = FALSE;

	if( (((grl_internal_cache_struct*)e1->value)->glyph == ((grl_internal_cache_struct*)e2 ->value)->glyph) &&
		(((grl_internal_cache_struct*)e1->value)->font_size == ((grl_internal_cache_struct*)e2 ->value)->font_size) &&
		(((grl_internal_cache_struct*)e1->value)->index == ((grl_internal_cache_struct*)e2 ->value)->index))
	{
		r = TRUE;
	}
	else
		r = FALSE;
	return r;
}


#ifdef HAS_OPENGLES2
#define MAX_SHADER_FORMATS 16 /* should be normally just one format */
#include <GLES2/gl2ext.h>

static SVGBoolean local_load_binary_shader(const GLuint *vertex_shader,
                    const GLuint *fragment_shader)
{
	SVGBoolean return_val = SVG_FALSE;
	GLint numFormats = 0;
	GLint listFormats[MAX_SHADER_FORMATS] = {0};
	SVGInt32 i;

	glGetIntegerv(GL_NUM_SHADER_BINARY_FORMATS,&numFormats);
	if ((numFormats != 0) && (numFormats < MAX_SHADER_FORMATS))
	{
        glGetIntegerv(GL_SHADER_BINARY_FORMATS,listFormats);
        for(i=0; i<numFormats; i++)
        {
            if(listFormats[i] == GL_SHADER_BINARY_VIV)
            {
                glShaderBinary(1, vertex_shader, GL_SHADER_BINARY_VIV, vertex, VERTEX_SIZE);
                if (glGetError() == GL_NO_ERROR)
                {
                    glShaderBinary(1, fragment_shader, GL_SHADER_BINARY_VIV, fragment, FRAGMENT_SIZE);

                    if (glGetError() == GL_NO_ERROR)
                    {
                        return_val = SVG_TRUE;
                    }
                }
            }
        }
	}

	if (return_val == SVG_FALSE)
	{
	    SVG_FNT_W("Failed to load binary shader."
	    		"Using source shader as fallback");
	}

	return return_val;
}
#endif

void grlftSetCachableGlyphCount(SVGFontContext *ctx,
	                            SVGUint32      glyph_count,
							    SVGFontInfo    *info)
{
	SVGFontFreeTypeRenderer   	*p_renderer        = (SVGFontFreeTypeRenderer*)ctx->p_fontRendererContext;
	SVGUint32 					glyphs_per_row     = (SVGUint32)sqrt(glyph_count);
	SVGUint32 					width_height       = (glyphs_per_row + MORE_CHARS_THEN_NEEDED ) * (SVGUint32)((SVGFloat)(p_renderer->font_size) * FILLED_PIXEL_RATIO);

	if (glyph_count > p_renderer->cachable_glyphs)
	{
		if (p_renderer->shared->glWallpaperWidth < p_renderer->max_texture_size)
        {
        	/* we need more space in texture, so remove previous wallpaper */
			if(width_height > p_renderer->shared->glWallpaperWidth)
        	{
				/* trace warning to user to explain why next frame is slow */
				if (NULL != p_renderer->row)
				{
				    SVG_FNT_F("GRL_CACHE_CLEANUP_NEEDED");
				}
				 /*delete hash table */
				UTIL_del_hsh(p_renderer->shared->hash_tbl_strk);
				UTIL_del_hsh(p_renderer->shared->hash_tbl_fill);

				local_clear_cache(p_renderer, SVG_FALSE);/*clear stroked cache*/
				local_clear_cache(p_renderer, SVG_TRUE);/*clear filled cache*/
				/* throw away the wallpapers */
				local_free_wallpapers( ctx );

				p_renderer->shared->cache_fill = NULL;
				p_renderer->shared->cache_strk = NULL;
				p_renderer->shared->cache_count_fill = 0;
				p_renderer->shared->cache_count_strk = 0;

				p_renderer->shared->cache_count = &p_renderer->shared->cache_count_fill;
				p_renderer->shared->cache = &p_renderer->shared->cache_fill;

				/* create new matching texture for caching */
				p_renderer->cachable_glyphs = glyph_count;

				p_renderer->shared->hash_tbl_strk		= UTIL_cre_hsh(p_renderer->cachable_glyphs, local_hash_cmp_fn );
				p_renderer->shared->hash_tbl_fill		= UTIL_cre_hsh(p_renderer->cachable_glyphs, local_hash_cmp_fn );
				if( p_renderer->shared->hash_tbl_strk == NULL )
				{
					SVG_FNT_F("SVG_OUT_OF_MEMORY in GRLXXSETCACHABLEGLYPHCOUNT");
				}
				if( p_renderer->shared->hash_tbl_fill == NULL )
				{
					SVG_FNT_F("SVG_OUT_OF_MEMORY in GRLXXSETCACHABLEGLYPHCOUNT");
				}
				/* create wallpaper texture */
				local_create_wallpapers( ctx , p_renderer->font_size );
				SVG_FNT_D("Create wallpaper for font glpyhs with dimension %d x %d"
			    		,p_renderer->shared->glWallpaperWidth
			    		,p_renderer->shared->glWallpaperWidth);
        	}
        	else
        	{
        		/* some more glyphs will fit into current texture without texture resizing */
        		p_renderer->cachable_glyphs = glyph_count;
        	}
        }
        else
        {
    		if (width_height < p_renderer->shared->glWallpaperWidth)
    		{
    			/* there will fit some more glyphs into current maximum texture */
    			p_renderer->cachable_glyphs = glyph_count;
    		}
    		else
    		{
				/* just do a trace to inform that cachable glyph count
				   could not be increased due to texture limitation */
			    SVG_FNT_W("could not increase number of cachable glyphs"
			    		" because max texture size is reached");
    		}
        }
	}

	if (NULL != info)
	{
		grlftGetFontInformation(ctx, info);
	}
}


SVGBoolean local_update_script_flag( SVGLangScript langscript,
                                     SVGBoolean   flag )
{
    SVGBoolean ret = SVG_FALSE;
    SVGUint8 script_loop = 0;
    while( NULL != gs_autolangdb.p_complxscriptlist[script_loop].scriptname)
    {
        if(gs_autolangdb.p_complxscriptlist[script_loop].script_tag == langscript)
        {
            gs_autolangdb.p_complxscriptlist[script_loop].flag = flag;
            ret = SVG_TRUE;
            break;
        }
        script_loop++;
    }
   return ret;
}

/*
 * if autolang_detect_enable is SVG_TRUE, enable globally auto language detection process
 * else enable the given script only
 */
SVGBoolean grlftEnableShapingEng(SVGLangScript langscript,
                                 SVGBoolean    autolang_detect_enable)
{
    SVGBoolean ret = SVG_TRUE;

    if( NULL != gs_autolangdb.p_complxscriptlist)
    {
        if(SVG_TRUE == autolang_detect_enable)
        {
            gs_autolangdb.autolangdetect = SVG_TRUE;
        }
        else if(SVG_SCRIPT_INVALID != langscript)
        {
            ret = local_update_script_flag(langscript,SVG_TRUE);
        }
     }
    else
    {
        /* AUTO_LANG_DETECTION is not enabled in svg_config.ini */
        ret = SVG_FALSE;
    }
    return ret;
}

/*
 * if autolang_detect_disable is SVG_TRUE, disable globally auto language detection process
 * else disable the given script only
 */

SVGBoolean grlftDisableShapingEng(SVGLangScript langscript,
                                  SVGBoolean    autolang_detect_disable)
{
    SVGBoolean ret = SVG_TRUE;
    if( NULL != gs_autolangdb.p_complxscriptlist)
    {
        if(SVG_TRUE == autolang_detect_disable)
        {
            gs_autolangdb.autolangdetect = SVG_FALSE;
        }
        else if(SVG_SCRIPT_INVALID != langscript)
        {
            ret = local_update_script_flag(langscript,SVG_FALSE);
        }
    }
    else
    {
        /* AUTO_LANG_DETECTION is not enabled in svg_config.ini */
        ret = SVG_FALSE;
    }
    return ret;
}

static SVGBoolean local_shpeng_get_autolang_config(void)
{
    return gs_autolangdb.autolangdetect;
}

void grlftReadShpEngConfig(void)
{
    Complex_Scripts_List compl_scriptslist[ ] = { SUPPORTED_COMPLEX_SCRIPTS_LIST };
    Simple_Scripts_List    simple_scriptlist[ ] = { SUPPORTED_SIMPLE_SCRIPTS_LIST };
    SVGChar  enable_flag_buf[4] = { 0};
    SVGInt32  a_length;
    SVGUint16  sizeof_scriptlist;

    a_length = GRL_get_string_config((SVGChar*)"SVG_AUTOLANG_DETECTION",
                    (SVGUint8*)&enable_flag_buf[0], 0);

    if((a_length > 0) && (4 > a_length))
    {
        GRL_get_string_config((SVGChar*)"SVG_AUTOLANG_DETECTION",
                    (SVGUint8*)&enable_flag_buf[0], a_length);

        if( 0 == strcasecmp(enable_flag_buf,"yes") )
        {
            sizeof_scriptlist = sizeof(compl_scriptslist);

            gs_autolangdb.p_complxscriptlist = (Complex_Scripts_List*)\
                            GRL_malloc_1D_resource(sizeof_scriptlist);
            if( NULL != gs_autolangdb.p_complxscriptlist)
            {
                memcpy(gs_autolangdb.p_complxscriptlist,compl_scriptslist,sizeof_scriptlist);

                 sizeof_scriptlist = sizeof(simple_scriptlist);
                 gs_autolangdb.p_simplescriptlist = (Simple_Scripts_List*)\
                                 GRL_malloc_1D_resource(sizeof_scriptlist);

                 if( NULL != gs_autolangdb.p_simplescriptlist)
                 {
                     memcpy(gs_autolangdb.p_simplescriptlist,simple_scriptlist,sizeof_scriptlist);
                     gs_autolangdb.autolangdetect = SVG_TRUE;
                  }
                  else
                  {
                        GRL_free_1D_resource(gs_autolangdb.p_complxscriptlist);
                        gs_autolangdb.p_complxscriptlist = NULL;
                  }
             }
        }
    }
    return;
}

static SVGError local_shpeng_set_langscript_rendrdirection(SVGFontFreeTypeRenderer *p_renderer,
                                                                                    SVGLangScript   langscript,
                                                                                    SVGRenderDirection  rendr_direction)
{
    SVGError    err = SVG_NO_ERROR;

    if( NULL != p_renderer->shared->p_shape_eng )
    {
        if(p_renderer->script_tag != langscript)
        {
                p_renderer->p_grl_shpeng_fp_table->\
                                    SetLangscript_rendrdirection( p_renderer->shared->p_shape_eng,
                                                                            langscript,
                                                                            rendr_direction);
        }
    }
    else
    {
        err = SVG_INVALID_OPERATION;
    }
    return err;
}

static SVGBoolean local_shpeng_find_scripttag ( get_char_info_struct*    p_charinfo,
                                                                  SVGUint32* p_scripttag,
                                                                 SVGUint32* p_rendrdirectag )
{
    SVGInt16    loop = 0;
//    SVGInt16   uni_start = 0;
//    SVGInt16   uni_end = 0;
    SVGBoolean  script_found = SVG_FALSE;
    SVGBoolean term_found = SVG_FALSE;
    SVGBoolean  searchinsimplelist= SVG_FALSE;
    SVGBoolean  next_charprocess_flag = SVG_TRUE;
    SVGUint8* text;
    SVGUint32 textLength;

    text = p_charinfo->theChar;
    textLength = p_charinfo->textlength;

    p_charinfo->theChar = local_decode_char(p_charinfo);

     while(SVG_FALSE == term_found)
     {
            if(SVG_FALSE == searchinsimplelist)
            {
               /* Finding the script tag and render direction */
               loop = 0;
               while( (SVG_FALSE == term_found) && (gs_autolangdb.p_complxscriptlist[loop].scriptname ))
               {
                  if((p_charinfo->decodedChar >= gs_autolangdb.p_complxscriptlist[loop].unicode_start) \
                              && (p_charinfo->decodedChar  <=  gs_autolangdb.p_complxscriptlist[loop].unicode_end))
                  {
                          /* Checking whether the script is allowed to process through shaping engine*/
                          if(gs_autolangdb.p_complxscriptlist[loop].flag)
                          {
                              *p_scripttag = gs_autolangdb.p_complxscriptlist[loop].script_tag;
                              *p_rendrdirectag = gs_autolangdb.p_complxscriptlist[loop].rendr_direc_tag;
                              script_found    =    SVG_TRUE;
                          }
                          term_found = SVG_TRUE;
                   }
                   ++loop;
              }
              if( SVG_TRUE == term_found)
                    break;

                  next_charprocess_flag = SVG_TRUE;
          }

/*
  * TODO
  * Need to collect and analyze the below things
  * >> further on some other simple scripts> extended glyphs,symbols,arabic forms ranges
  * >> how these glyphs are aligned/used in the strings
  *  will enable this optimization after preparing the full simple list
  */
#if 0
            /*
              * if the characer unicode is not in the complex script list, search in simple list
              * if it gets hit in the simple script list then the next characer will be searched directly
              * on the simple list with same unicode range first.
              * The search will be back to complex script list when it is not found in simple list
              */
           {
                loop = 0;
                while(gs_autolangdb.p_simplescriptlist[loop].unicode_end)
                {
                     if((p_charinfo->decodedChar >= gs_autolangdb.p_simplescriptlist[loop].unicode_start) \
                                && (p_charinfo->decodedChar  <=  gs_autolangdb.p_simplescriptlist[loop].unicode_end))
                     {
                           next_charprocess_flag = SVG_TRUE;
                           searchinsimplelist        = SVG_TRUE;

                           if( loop != 0)
                           {
                               uni_start = gs_autolangdb.p_simplescriptlist[loop].unicode_start;
                               uni_end  = gs_autolangdb.p_simplescriptlist[loop].unicode_end;

                               gs_autolangdb.p_simplescriptlist[loop].unicode_start =  \
                                                   gs_autolangdb.p_simplescriptlist[0].unicode_start;
                               gs_autolangdb.p_simplescriptlist[loop].unicode_end =  \
                                                   gs_autolangdb.p_simplescriptlist[0].unicode_end;
                             gs_autolangdb.p_simplescriptlist[0].unicode_start =  uni_start;
                             gs_autolangdb.p_simplescriptlist[0].unicode_end =  uni_end;
                         }
                           break;
                     }
                     ++loop;
                }
                if(!gs_autolangdb.p_simplescriptlist[loop].unicode_end)
                {
                    searchinsimplelist = SVG_FALSE;
                    next_charprocess_flag = SVG_FALSE;
                }
          }
#endif

        if(SVG_TRUE == next_charprocess_flag)
        {
             p_charinfo->theChar = &p_charinfo->theChar[1];
             if(!(p_charinfo->theChar < &text[textLength]))
             {
                term_found = SVG_TRUE;
             }
             else
             {
                   p_charinfo->theChar = local_decode_char(p_charinfo);
             }
        }
    }
    return script_found;
}
